package ru.yandex.direct.common.jooqmapper;

import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.TableField;

/**
 * Класс отвечает за модификацию модели на основе данных из базы.
 * <p>
 * Содержит: поле базы, значение которого получает из передаваемой записи {@link Record},
 * функцию-конвертер, при необходимости преобразующую значение из базы во внутренний формат,
 * и функцию {@link BiConsumer}, принимающую экземпляр модели и сконвертированное значение
 * из базы, которая выставляет значение непосредственно в модели.
 * <p>
 * Хорошим стилем является выставлять одно поле модели на основе одного поля в базе.
 * Если на основе одного поля в базе необходимо выставить несколько полей модели,
 * то лучше использовать несколько экземпляров данного класса, регистрируя
 * в {@link OldJooqMapper} несколько экземпляров {@link FieldMapper} для одного поля БД.
 *
 * @param <R> тип записи, к которой относится поле таблицы БД
 * @param <M> тип модели
 * @param <V> тип значения в базе
 * @param <X> тип значения в модели (тип, возвращаемый конвертером и принимаемый финкцией "сеттером")
 */
@ParametersAreNonnullByDefault
public class FieldReader<R extends Record, M, V, X> {

    /**
     * Поле в базе, значение которого читается из передаваемой записи {@link Record}
     */
    private final TableField<R, V> tableField;

    /**
     * Функция, принимающая модель и сконвертированное значение поля из базы,
     * и непосредственно модифицирующая модель
     */
    private BiConsumer<M, X> setter;

    /**
     * Функция-конвертер, принимающая на вход значение поля из базы и преобразующая его во внутренний формат
     */
    private Function<V, X> converter;

    /**
     * Дефолтное значение, устанавливаемое в модели, когда
     * извлеченное из базы значение равно {@code null} (после применения конвертера).
     */
    private X defaultValue;

    FieldReader(TableField<R, V> tableField) {
        this.tableField = tableField;
    }

    void apply(M model, Record record) {
        X value;
        try {
            value = converter.apply(record.getValue(tableField));
        } catch (NullPointerException e) {
            throw new JooqMapperException("NullPointer while applying converter for " + tableField.getName(), e);
        }
        if (value == null) {
            value = defaultValue;
        }
        setter.accept(model, value);
    }

    /**
     * Устанавливает функцию, которая выставляет в модели значение, полученное из записи в БД.
     * <p>
     * Функция принимает уже сконвертированное значение из типа данных поля в базе во внутренний тип данных.
     * Потенциально функция на основе полученного значения может производить с моделью любые действия,
     * например, выставить несколько полей одновременно. Однако такой подход не рекомендуется, так как
     * усложняет понимание кода. Чтобы установить несколько полей модели на основе одного поля в базе,
     * создавайте для каждого поля собственные мапперы и регистрируйте в {@link OldJooqMapper}. При регистрации
     * нескольких мапперов для чтения из одного поля в базе, может быть полезным запретить им запись в базу
     * с помощью метода {@link FieldMapper#disableWritingToDb}
     *
     * @param setter функция, принимающая экземпляр модели и сконвертированное во внутренний формат значение
     *               из базы, которая должна выставить поля модели на основе этого значения.
     * @return возвращает себя
     */
    FieldReader<R, M, V, X> by(BiConsumer<M, X> setter) {
        this.setter = setter;
        return this;
    }

    /**
     * Устанавливает функцию-конвертер, которая принимает значение, полученное из базы,
     * и возвращает значение для записи в модель. В основном используется для приведения/конвертации типов,
     * когда тип данных в модели не соответствует типу данных в базе.
     *
     * @param converter функция-конвертер
     * @return возвращает себя
     */
    FieldReader<R, M, V, X> convertBy(Function<V, X> converter) {
        this.converter = converter;
        return this;
    }

    /**
     * Устанавливает дефолтное значение, которое будет установлено в модели в том случае,
     * если извлеченное из базы значение равно {@code null} (после применения конвертера).
     *
     * @param defaultValue дефолтное значение для модели
     * @return возвращает себя
     */
    FieldReader<R, M, V, X> withDefaultValue(X defaultValue) {
        this.defaultValue = defaultValue;
        return this;
    }
}
