package ru.yandex.direct.common.jooqmapper;

import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.TableField;

import static org.jooq.impl.DSL.defaultValue;

/**
 * Класс отвечает за запись значений из модели в базу данных
 * посредством Jooq-билдеров.
 * <p>
 * Содержит функцию, достающую значение из модели, функцию-конвертер, которая
 * при необходимости конвертирует значение из модели в формат базы данных,
 * а так же дефолтное значение, которое может быть записано в случае, когда
 * значение из модели равно {@code null}, и флаг, позволяющий использовать
 * дефолтное значение уровня базы данных.
 *
 * @param <R> тип записи, определяет таблицу, с которым может работать экземпляр класса
 * @param <M> тип модели
 * @param <V> тип значения в базе
 * @param <X> тип значения в модели
 */
@ParametersAreNonnullByDefault
public class FieldWriter<R extends Record, M, V, X> {

    /**
     * Поле таблицы, значение которого выставляется в Jooq-билдере для записи в базу
     */
    private final TableField<R, V> tableField;

    /**
     * Функция, принимающая на вход модель и возвращающая некоторое значение
     */
    private Function<M, X> getter;

    /**
     * Функция-конвертер, преобразующая значение из внутреннего формата в формат базы данных
     */
    private Function<X, V> converter;

    /**
     * Дефолтное значение, которое может быть выставлено в базе
     * в случае, если значение из модели равно {@code null}
     */
    private X defaultValue;

    /**
     * Флаг, показывающий, использовать или нет дефолтное значение
     * уровня базы данных, если значение из модели равно {@code null}
     */
    private boolean databaseDefault;

    FieldWriter(TableField<R, V> tableField) {
        this.tableField = tableField;
    }

    /**
     * Устанавливает функцию, получающую из модели значение для записи в поле таблицы в базе.
     * Тип получаемого значения может отличаться от типа поля в базе, для установки конвертера
     * используйте {@link #convertBy(Function)}
     *
     * @param getter функция "геттер", получающая из модели значение для записи в поле таблицы в базе
     * @return возвращает себя
     */
    FieldWriter<R, M, V, X> by(Function<M, X> getter) {
        this.getter = getter;
        return this;
    }

    /**
     * Устанавливает функцию-конвертер, которая принимает значение, полученное из модели,
     * и возвращает значение для записи в базу. В основном используется для приведения/конвертации типов,
     * когда тип данных в модели не соответствует типу данных в базе.
     *
     * @param converter функция-конвертер
     * @return возвращает себя
     */
    FieldWriter<R, M, V, X> convertBy(Function<X, V> converter) {
        this.converter = converter;
        return this;
    }

    /**
     * Устанавливает значение, которое будет записано в базу в случае,
     * если из модели извлечено значение {@code null}.
     *
     * @param defaultValue дефолтное значение для записи в базу
     * @return возвращает себя
     */
    FieldWriter<R, M, V, X> withDefaultValue(X defaultValue) {
        this.defaultValue = defaultValue;
        return this;
    }

    /**
     * Устанавливает поведение, при котором, если из модели извлечено значение {@code null},
     * то в базу будет записано дефолтное значение, определенное в самой таблице.
     *
     * @return возвращает себя
     */
    FieldWriter<R, M, V, X> withDatabaseDefault(boolean databaseDefault) {
        this.databaseDefault = databaseDefault;
        return this;
    }

    /**
     * @return возвращает либо V, либо Field<V> (если берется дефолтное значение из базы)
     */
    public Object getDbFieldValue(M model) {
        V value = pullValue(model);
        if (value == null && databaseDefault) {
            return defaultValue(tableField);
        }
        return value;
    }


    private V pullValue(M model) {
        V value;
        try {
            value = converter.apply(getter.apply(model));
        } catch (NullPointerException e) {
            throw new JooqMapperException("NullPointer while applying converter for " + tableField.getName(), e);
        }

        if (value == null && !databaseDefault) {
            value = converter.apply(defaultValue);
        }
        return value;
    }
}
