package ru.yandex.direct.common.jooqmapper;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqModelToDbMapper;
import ru.yandex.direct.jooqmapper.read.JooqReader;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.Collectors.nullFriendlyMapCollector;

/**
 * Класс для описания правил записи полей модели в базу данных
 * и чтения полей модели из базы данных с помощью Jooq.
 * <p>
 * Инстанцируется с помощью {@link OldJooqMapperBuilder}.
 * При построении регистрируются мапперы {@link FieldMapper} для записи полей модели в базу
 * и чтения из базы.
 * <p>
 * После этого может многократно использоваться для:<ul>
 * <li>
 * получения экземпляров модели на основе записи {@link Record},
 * см. {@link #fromDb(M, Record)}. Поддерживается частичное заполнение модели,
 * т.е. в передаваемом {@link Record} могут присутствовать не все поля,
 * для которых зарегистрированы мапперы.
 * </li>
 * <li>
 * заполнения Jooq-билдеров данными из модели для последующей записи в базу данных.
 * Для упрощения этой операции можно использовать {InsertHelper}.
 * </li>
 * </ul>
 * <p>
 * Если при чтении модели из базы данных всегда известно, какой тип модели
 * будет создаваться, то можно использовать {@link OldJooqMapperWithSupplier},
 * имеющий упрощенный метод {@link OldJooqMapperWithSupplier#fromDb(Record)}.
 * <p>
 * Пример использования с {@link ModelProperty}
 * (Person.ID, Person.NAME, Person.GENDER - это ModelProperty):
 * <pre> {@code
 *
 *  // фабричные методы для удобства создания мапперов
 *  import static ru.yandex.direct.common.jooqbus.FieldMapperFactory.convertibleField;
 *  import static ru.yandex.direct.common.jooqbus.FieldMapperFactory.field;
 *
 *  public class PersonRepository {
 *
 *      // ...
 *
 *      private final OldJooqMapper<Person> personMapper;
 *
 *      public PersonRepository() {
 *          personMapper = new OldJooqMapperBuilder<>(Person::new)
 *                  .map(field(PERSONS.ID, Person.ID))      // строго записывает то, что в модели, даже если там null
 *                  .map(field(PERSONS.NAME, Person.NAME)
 *                          .withDefaultValueForDb(""))     // если поле модели == null, записывает пустую строку
 *
 *                  // с конвертацией Jooq-енама во внутренний формат и обратно
 *                  .map(convertibleField(PERSONS.GENDER, Person.GENDER)
 *                          .convertToDbBy(Mappings::genderToDb)
 *                          .convertFromDbBy(Mappings::genderFromDb))
 *                  .build()
 *      }
 *
 *      public List<Person> getPersons(List<Long> ids) {
 *          return jooqDslContext()
 *                  .select(personMapper.getFieldsToRead())
 *                  .from(PERSONS)
 *                  .where(PERSONS.ID.in(ids))
 *                  .fetch()
 *                  .map(rec -> personMapper.fromDb(new Person(), rec));
 *      }
 *
 *      public void addPersons(List<Person> persons) {
 *          new InsertHelper<>(jooqDslContext(), PERSONS)
 *                  .addAll(personMapper, persons)
 *                  .executeIfRecordsAdded();
 *      }
 *  }
 * }</pre>
 *
 * @param <M> тип модели
 * @see OldJooqMapperWithSupplier
 * @deprecated use {@link JooqMapper}
 * or {@link JooqMapperWithSupplier}
 * or {@link JooqReader}
 * or {@link JooqReaderWithSupplier}
 * or {@link JooqWriter}
 * instead.
 */
@Deprecated
@ParametersAreNonnullByDefault
public class OldJooqMapper<M> implements JooqModelToDbMapper<M> {

    /**
     * Мапперы, управляющие чтением из базы и записью в базу.
     */
    private final List<FieldMapper<?, ? super M, ?, ?>> mappers;

    /**
     * Мапа для быстрого доступа к списку мапперов, предназначенных для определенного поля базы.
     * Используется при чтении из базы, так как чтение принимает на вход Record с
     * любым набором полей и итерируется по этому набору.
     */
    private final Map<TableField<?, ?>, List<FieldMapper<?, ? super M, ?, ?>>> mappersByField;

    /**
     * Мапа для быстрого доступа к списку мапперов по типу таблицы в базе, для которой они предназначены.
     * Используется при записи в базу, так как один JooqBus поддерживает мапперы сразу для нескольких таблиц,
     * а запись осуществляется в одну конкретную таблицу.
     */
    private final Map<Table, List<FieldMapper<?, ? super M, ?, ?>>> mappersByTable;

    /**
     * {@link Set} с теми полями, которые данный {@link OldJooqMapper} умеет читать
     */
    private final Set<TableField<?, ?>> fieldsWithReadingEnabled;

    /**
     * Функция, которую нужно применить к модели, после того как она была построена на основе полученных из базы
     * записей
     */
    private final Function<M, M> modelPostprocessor;

    OldJooqMapper(List<FieldMapper<?, ? super M, ?, ?>> mappers, @Nullable Function<M, M> modelPostprocessor) {
        this.modelPostprocessor = modelPostprocessor;
        this.mappers = ImmutableList.copyOf(mappers);
        this.fieldsWithReadingEnabled = ImmutableSet.copyOf(StreamEx.of(mappers)
                .filter(FieldMapper::isReadingFromDbEnabled)
                .map(FieldMapper::getTableField)
                .map(field -> (TableField<?, ?>) field)
                .toList());
        this.mappersByField = ImmutableMap.copyOf(mappers
                .stream()
                .collect(groupingBy(FieldMapper::getTableField, mapping(identity(), toList()))));
        this.mappersByTable = ImmutableMap.copyOf(mappers
                .stream()
                .collect(groupingBy(FieldMapper::getTable, mapping(identity(), toList()))));
    }

    /**
     * Заполняет переданный экземпляр модели данными из переданной записи в БД,
     * используя предварительно заданные мапперы.
     * <p>
     * Поддерживает частичное заполнение модели, то есть в переданной записи могут находиться
     * не все поля таблицы, для которых зарегистрированы мапперы. Соответственно, будут применены
     * только те мапперы, которые читают поля, присутствующие в записи.
     * <p>
     * Поддерживает заполнение из нескольких таблиц, т.е. в переданной записи могут одновременно
     * находиться значения из разных таблиц.
     *
     * @param record запись в БД, из которой необходимо заполнить модель
     * @return экземпляр модели, заполненный из переданной записи в БД
     */
    public M fromDb(M model, Record record) {

        boolean readCompleted = false;

        for (Field f : record.fields()) {
            /*
                Для извлеченного из базы поля setter может быть незарегистрирован,
                например, когда строка таблицы достается целиком, но некоторые
                поля игнорируются, так как устарели.
             */
            List<FieldMapper<?, ? super M, ?, ?>> fieldMappers = mappersByField.get(f);
            if (fieldMappers == null) {
                continue;
            }

            for (FieldMapper<?, ? super M, ?, ?> mapper : fieldMappers) {
                if (mapper.isReadingFromDbEnabled()) {
                    readCompleted = true;
                    mapper.fromDb(model, record);
                }
            }
        }
        checkState(readCompleted, "there is no registered mappers for reading fields in specified Jooq record");
        if (modelPostprocessor != null) {
            model = modelPostprocessor.apply(model);
        }
        return model;
    }

    /**
     * Возвращает список всех полей, для которых заданы мапперы с включенным чтением из базы данных.
     * Поля могут принадлежать разным таблицам.
     */
    @SuppressWarnings("squid:S1452")
    public Collection<TableField<?, ?>> getFieldsToRead() {
        return fieldsWithReadingEnabled;
    }

    @SuppressWarnings("squid:S1452")
    Stream<FieldMapper<?, ? super M, ?, ?>> mappers() {
        return this.mappers.stream();
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(M model, Table<R> table) {
        List<FieldMapper<?, ? super M, ?, ?>> tableMappers = mappersByTable.get(table);
        checkState(tableMappers != null, "there is no registered mappers for {}", table);
        //noinspection unchecked приведение типа безапасно, т.к. предварительно делается filter.
        return StreamEx.of(tableMappers)
                .filter(FieldMapper::isWritingToDbEnabled)
                .mapToEntry(FieldMapper::getTableField, v -> v.getDbFieldValue(model))
                .filterKeys(tableField -> tableField.getTable().equals(table))
                .mapKeys(tableField -> (TableField<R, ?>) tableField)
                // стандартный коллектор toMap не выносит null значений (кидает NPE)
                .collect(nullFriendlyMapCollector());
    }
}
