package ru.yandex.direct.common.jooqmapperex;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Set;
import java.util.function.Function;

import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.common.jooqmapper.FieldMapper;
import ru.yandex.direct.common.jooqmapper.FieldMapperFactory;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.common.util.RepositoryUtils.nullSafeWrapper;

public class FieldMapperFactoryEx {
    private FieldMapperFactoryEx() {
    }

    /**
     * Возвращает маппер для даты. В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и конвертеры даты в формат базы данных и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для даты
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, LocalDate, LocalDate> dateField(
            TableField<R, LocalDate> tableField, ModelProperty<M, LocalDate> property) {
        return FieldMapperFactory.field(tableField, property);
    }

    /**
     * Возвращает маппер для таймштампа. В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и конвертеры таймштампа в формат базы данных и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для таймштампа
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, LocalDateTime, LocalDateTime> timestampField(
            TableField<R, LocalDateTime> tableField, ModelProperty<M, LocalDateTime> property) {
        return FieldMapperFactory.field(tableField, property);
    }

    /**
     * Возвращает маппер для boolean поля. В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и конвертеры boolean в long для базы данных и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для boolean поля
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, Long, Boolean> booleanField(
            TableField<R, Long> tableField, ModelProperty<M, Boolean> property) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(RepositoryUtils::booleanToLong)
                .convertFromDbBy(RepositoryUtils::booleanFromLong);
    }

    /**
     * Возвращает маппер для boolean поля. В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и конвертеры boolean в long для базы данных и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для boolean поля
     */
    public static <R extends Record, M extends Model, E extends Enum<E>> FieldMapper<R, M, E, Boolean> yesNoField(
            TableField<R, E> tableField, ModelProperty<M, Boolean> property, Class<E> cls) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(mv -> RepositoryUtils.booleanToYesNo(mv, cls))
                .convertFromDbBy(RepositoryUtils::booleanFromYesNo);
    }

    /**
     * Возвращает маппер для Integer. В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и конвертеры инта в формат базы данных (Long) и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для Integer
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, Long, Integer> integerField(
            TableField<R, Long> tableField, ModelProperty<M, Integer> property) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(RepositoryUtils::intToLong)
                .convertFromDbBy(RepositoryUtils::intFromLong);
    }

    /**
     * Возвращает маппер для {@link Percent}. В возвращаемом маппере уже установлены геттер и сеттер на основе
     * переданной {@link ModelProperty} и конвертеры процента в формат базы данных (BigDecimal) и обратно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для {@link Percent}
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, BigDecimal, Percent> percentField(
            TableField<R, BigDecimal> tableField, ModelProperty<M, Percent> property) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(RepositoryUtils::percentToBigInteger)
                .convertFromDbBy(RepositoryUtils::percentFromBigInteger);
    }

    public static <R extends Record, M extends Model> FieldMapper<R, M, Long, ClientId> clientIdField(
            TableField<R, Long> tableField, ModelProperty<M, ClientId> property) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(nullSafeWrapper(ClientId::asLong))
                .convertFromDbBy(nullSafeWrapper(ClientId::fromLong));
    }

    /**
     * Возвращает маппер множества в базе данных, которое представлено множеством значений енама в модели
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param enumCls    объект, описывающий класс енама
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @param <E>        тип енама
     * @return готовый к применению экземпляр {@link FieldMapper} для сета енамов
     */
    public static <R extends Record, M extends Model, E extends Enum<E>> FieldMapper<R, M, String, Set<E>> enumSetField(
            TableField<R, String> tableField, ModelProperty<M, Set<E>> property, Class<E> enumCls) {
        return enumSetField(tableField, property, str -> Enum.valueOf(enumCls, str), Object::toString);
    }

    /**
     * Возвращает маппер множества в базе данных, которое представлено множеством значений енама в модели
     * Для превращения строк в множестве в БД в значения енама и обратно используются переданные функции-мепперы
     * <p>
     * TODO реализация не завязана ни на что в enum, так что стоит оторвать E extends Enum&lt;E&gt; и переименовать
     * функцию; при этом надо обратить внимание, что рядом уже есть setField, который делает принципиально другое
     *
     * @param tableField   поле базы данных
     * @param property     поле модели
     * @param fromDbMapper как превращать строки в множестве в БД в значения енама
     * @param toDbMapper   как превращать значения енама в строки в БД
     * @param <R>          тип записи, к которой относится поле таблицы БД
     * @param <M>          тип модели
     * @param <E>          тип енама
     * @return готовый к применению экземпляр {@link FieldMapper} для сета енамов
     */
    public static <R extends Record, M extends Model, E extends Enum<E>> FieldMapper<R, M, String, Set<E>> enumSetField(
            TableField<R, String> tableField, ModelProperty<M, Set<E>> property,
            Function<String, E> fromDbMapper, Function<E, String> toDbMapper) {
        return FieldMapperFactory.convertibleField(tableField, property)
                .convertToDbBy(set -> RepositoryUtils.setToDb(set, toDbMapper))
                .convertFromDbBy(s -> RepositoryUtils.setFromDb(s, fromDbMapper));
    }
}
