package ru.yandex.direct.common.jooqmapperex;

import java.math.BigDecimal;
import java.util.Set;
import java.util.function.Function;

import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.ReaderWriterBuilder;
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class ReaderWriterBuildersEx {
    private ReaderWriterBuildersEx() {
    }

    public static <M extends Model, X extends Record> ReaderWriterBuilder<M, Boolean, X, Long> booleanProperty(
            ModelProperty<? super M, Boolean> modelProperty,
            TableField<X, Long> field) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong);
    }

    public static <M extends Model, X extends Record, E extends Enum<E>> ReaderWriterBuilder<M, Boolean, X, E>
    booleanProperty(
            ModelProperty<? super M, Boolean> modelProperty,
            TableField<X, E> field,
            Class<E> yesNoEnumClass) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field, RepositoryUtils::booleanFromYesNo,
                boolValue -> RepositoryUtils.booleanToYesNo(boolValue, yesNoEnumClass));
    }

    public static <M extends Model, X extends Record> ReaderWriterBuilder<M, Integer, X, Long> integerProperty(
            ModelProperty<? super M, Integer> modelProperty,
            TableField<X, Long> field) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                RepositoryUtils::intFromLong, RepositoryUtils::intToLong);
    }

    public static <M extends Model, X extends Record> ReaderWriterBuilder<M, Percent, X, BigDecimal> percentProperty(
            ModelProperty<? super M, Percent> modelProperty,
            TableField<X, BigDecimal> field) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                RepositoryUtils::percentFromBigInteger, RepositoryUtils::percentToBigInteger);
    }

    public static <M extends Model, X extends Record> ReaderWriterBuilder<M, ClientId, X, Long> clientIdProperty(
            ModelProperty<? super M, ClientId> modelProperty,
            TableField<X, Long> field) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                ClientId::fromNullableLong, clientId -> ifNotNull(clientId, ClientId::asLong));
    }

    public static <M extends Model, X extends Record, E extends Enum<E>>
    ReaderWriterBuilder<M, Set<E>, X, String> convertibleEnumSet(
            ModelProperty<? super M, Set<E>> modelProperty, TableField<X, String> field,
            Function<String, E> fromDbMapper, Function<E, String> toDbMapper
    ) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                str -> RepositoryUtils.setFromDb(str, fromDbMapper),
                set -> RepositoryUtils.setToDb(set, toDbMapper));
    }

    public static <M extends Model, X extends Record, E extends Enum<E>>
    ReaderWriterBuilder<M, Set<E>, X, String> convertibleEnumSet(
            ModelProperty<? super M, Set<E>> modelProperty,
            TableField<X, String> field, Class<E> enumType
    ) {
        return convertibleEnumSet(modelProperty, field,
                s -> Enum.valueOf(enumType, s), Object::toString);
    }

    public static <M extends Model, X extends Record> ReaderWriterBuilder<M, Set<String>, X, String> stringSetProperty(
            ModelProperty<? super M, Set<String>> modelProperty,
            TableField<X, String> dbField
    ) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, dbField,
                dbStr -> RepositoryUtils.setFromDb(dbStr, Function.identity()),
                set -> RepositoryUtils.setToDb(set, Function.identity())
        );
    }

    /**
     * Возвращает билдер для объекта, хранящегося в базе в виде сжатой json-сериализации в поле типа Mediumblob.
     * Значения равные {@code null} де/сериализует as is.
     *
     * @param modelProperty поле модели
     * @param field         поле базы данных
     * @param typeClass     класс поля модели (нужен для десериализации)
     * @param <M>           тип модели
     * @param <X>           тип записи, к которой относится пое таблицы БД
     * @param <T>           тип поля модели
     * @return готовый к применению билдер
     * @implNote при сохранении данных выкинет {@link IllegalArgumentException} если сжатые данные не поместятся
     */
    public static <M extends Model, T, X extends Record>
    ReaderWriterBuilder<M, T, X, byte[]> compressedMediumblobJsonProperty(
            ModelProperty<? super M, T> modelProperty, TableField<X, byte[]> field, Class<T> typeClass
    ) {
        return ReaderWriterBuilders.convertibleProperty(modelProperty, field,
                f -> RepositoryUtils.objectFromCompressedJsonDb(f, typeClass),
                RepositoryUtils::toCompressedJsonMediumblobDb);
    }
}
