package ru.yandex.direct.common.jooqmapperex.read;

import org.jooq.Field;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.jooqmapper.read.Reader1Builder;
import ru.yandex.direct.jooqmapper.read.ReaderBuilders;

import static org.apache.http.util.TextUtils.isEmpty;
import static ru.yandex.direct.common.util.RepositoryUtils.intFromLong;
import static ru.yandex.direct.common.util.RepositoryUtils.zeroToNull;

public class ReaderBuildersEx {
    private static final String TRUE = "yes";

    private ReaderBuildersEx() {
    }

    /**
     * Получить Boolean из "yes" строки, игнорируя регистр.
     */
    public static Reader1Builder<String, Boolean> fromYesNoFieldToBoolean(Field<String> field) {
        return ReaderBuilders.fromField(field).by(value -> value != null ? value.equalsIgnoreCase(TRUE) : null);
    }

    /**
     * Получить Boolean из Enum.name() "yes".
     */
    public static <T extends Enum<T>> Reader1Builder<T, Boolean> fromYesNoEnumFieldToBoolean(Field<T> field) {
        return ReaderBuilders.fromField(field).by(RepositoryUtils::booleanFromYesNo);
    }

    /**
     * Получить Boolean из Enum.name() "yes".
     * false, если значение null
     */
    public static <T extends Enum<T>> Reader1Builder<T, Boolean> fromNullableYesNoEnumFieldToBoolean(Field<T> field) {
        return ReaderBuilders.fromField(field).by(t -> t != null && t.name().equalsIgnoreCase(TRUE));
    }

    /**
     * Получить Boolean из Long
     */
    public static Reader1Builder<Long, Boolean> fromLongFieldToBoolean(Field<Long> field) {
        return ReaderBuilders.fromField(field).by(RepositoryUtils::booleanFromLong);
    }

    /**
     * Получить Integer из Long
     */
    public static Reader1Builder<Long, Integer> fromLongFieldToInteger(Field<Long> field) {
        return ReaderBuilders.fromField(field).by(RepositoryUtils::intFromLong);
    }

    /**
     * Получить Integer из Long, значениe 0 отображаем в null
     */
    public static Reader1Builder<Long, Integer> fromLongFieldToIntegerWithZeroNullification(Field<Long> field) {
        return ReaderBuilders.fromField(field).by(val -> zeroToNull(intFromLong(val)));
    }

    /**
     * Получить Enum из строки
     */
    public static <T extends Enum<T>> Reader1Builder<String, T> fromStringFieldToEnum(Field<String> field,
                                                                                      Class<T> enumCls) {
        return ReaderBuilders.fromField(field).by(value -> !isEmpty(value) ? Enum.valueOf(enumCls,
                value.toUpperCase()) : null);
    }

    /**
     * Получить Enum из строки с возможностью задать дефолтное значение
     */
    public static <T extends Enum<T>> Reader1Builder<String, T> fromStringFieldToEnum(
            Field<String> field, Class<T> enumCls, T defaultValue) {
        return ReaderBuilders.fromField(field).by(value -> !isEmpty(value) ? Enum.valueOf(enumCls,
                value.toUpperCase()) : defaultValue);
    }
}
