package ru.yandex.direct.common.liveresource.yav

import com.google.common.cache.CacheBuilder
import com.google.common.cache.LoadingCache
import org.apache.http.client.utils.URLEncodedUtils
import org.springframework.stereotype.Component
import ru.yandex.direct.liveresource.LiveResource
import ru.yandex.direct.liveresource.provider.LiveResourceProvider
import ru.yandex.direct.yav.client.YavClient
import java.net.URI
import java.time.Duration
import java.util.concurrent.ConcurrentHashMap

@Component
class YavLiveResourceProvider(private val yavClient: YavClient) : LiveResourceProvider {

    private val caches = ConcurrentHashMap<Duration, LoadingCache<String, Map<String, String>>>()

    override fun getScheme() = "yav"

    /**
     * yav://secretUuidOrVersion[?refreshRate=durationInSeconds]#keyName
     * получит ключ keyName из секрета secretUuidOrVersion
     * секрет кешируется и обновляется раз в durationInSeconds секунд.
     * Если значение не указано, то используется значение по-умолчанию DEFAULT_REFRESH_RATE_DURATION_SECONDS
     */
    override fun get(uri: URI): LiveResource {
        val queryParams = URLEncodedUtils.parse(uri, Charsets.UTF_8.toString())

        var refreshRate = Duration.ofSeconds(DEFAULT_REFRESH_RATE_DURATION_SECONDS)
        for (keyValue in queryParams) {
            when (keyValue.name) {
                REFRESH_RATE_KEY_NAME -> {
                    refreshRate = Duration.ofSeconds(keyValue.value.toLong())
                }
            }
        }

        val refreshingCache = caches.computeIfAbsent(refreshRate) { duration ->
            CacheBuilder.newBuilder()
                .maximumSize(100)
                .refreshAfterWrite(duration)
                .build(YavCacheLoader(yavClient))
        }
        return YavLiveResource(uri.authority, uri.fragment, refreshingCache)
    }

    private companion object {
        const val REFRESH_RATE_KEY_NAME = "refreshRate"
        const val DEFAULT_REFRESH_RATE_DURATION_SECONDS = 10L
    }
}

