package ru.yandex.direct.common.liveresource.zookeeper;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceEvent;
import ru.yandex.direct.liveresource.LiveResourceListener;
import ru.yandex.direct.liveresource.LiveResourceReadException;
import ru.yandex.direct.liveresource.LiveResourceWatcher;
import ru.yandex.direct.utils.ThreadUtils;

@ParametersAreNonnullByDefault
public class ZookeeperLiveResourceWatcher implements LiveResourceWatcher, Watcher {
    private static final Logger logger = LoggerFactory.getLogger(ZookeeperLiveResourceWatcher.class);

    private static final long RETRY_DELAY = Duration.ofSeconds(5).toMillis(); // 5 seconds

    private String previousContent;

    private final ZookeeperLiveResource liveResource;
    private final CuratorFrameworkProvider curatorFrameworkProvider;
    private final TaskScheduler taskScheduler;
    private final List<LiveResourceListener> listeners = new CopyOnWriteArrayList<>();

    private volatile boolean isClosed;
    private ScheduledFuture<?> scheduledRetry;

    public ZookeeperLiveResourceWatcher(ZookeeperLiveResource liveResource,
                                        CuratorFrameworkProvider curatorFrameworkProvider,
                                        TaskScheduler taskScheduler) {
        this.liveResource = liveResource;
        this.curatorFrameworkProvider = curatorFrameworkProvider;
        this.taskScheduler = taskScheduler;
    }

    public void addListener(LiveResourceListener listener) {
        listeners.add(listener);
    }

    public String watch() {
        try {
            previousContent = resetWatch();
            return previousContent;
        } catch (Exception e) {
            ThreadUtils.checkInterrupted(e);
            throw new LiveResourceReadException("Could not get initial zookeeper live resource value for location " +
                    liveResource.getLocation(), e);
        }
    }

    private String resetWatch() throws Exception {
        byte[] bytes = curatorFrameworkProvider.getDefaultCurator().getData()
                .usingWatcher(this).forPath(liveResource.getLocation());
        return new String(bytes, StandardCharsets.UTF_8);
    }

    private void resetWatchOrRetry() {
        try {
            checkContent(resetWatch());
        } catch (KeeperException.NoNodeException e) {
            logger.error("No zookeeper node for path {}", liveResource.getLocation());
            scheduleRetry();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (Exception e) {
            logger.error("Exception setting a watch for zookeeper node {}", liveResource.getLocation(), e);
            scheduleRetry();
        }
    }

    private void scheduleRetry() {
        scheduledRetry = taskScheduler.schedule(this::resetWatchOrRetry, Instant.now().plusMillis(RETRY_DELAY));
    }

    @Override
    public void process(WatchedEvent event) {
        if (!isClosed) {
            resetWatchOrRetry();
        }
    }

    private void checkContent(String currentContent) {
        if (!Objects.equals(previousContent, currentContent)) {
            previousContent = currentContent;
            notifyListeners(currentContent);
        }
    }

    private void notifyListeners(String currentContent) {
        LiveResourceEvent event = new LiveResourceEvent(currentContent);
        for (LiveResourceListener listener : listeners) {
            try {
                listener.update(event);
            } catch (RuntimeException ex) {
                logger.error("Listener {} failed to process LiveResourceEvent for location {}",
                        listener, liveResource.getLocation(), ex);
            }
        }
    }

    @Override
    public void close() {
        isClosed = true;
        if (scheduledRetry != null) {
            scheduledRetry.cancel(true);
        }
        try {
            curatorFrameworkProvider.getDefaultCurator().watches().remove(this).locally()
                    .forPath(liveResource.getLocation());
        } catch (Exception e) {
            ThreadUtils.checkInterrupted(e);
            logger.warn("Failed to remove zookeeper live resource watch", e);
        }
    }

    @Override
    public LiveResource getLiveResource() {
        return liveResource;
    }
}
