package ru.yandex.direct.common.log;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.common.log.container.LogEntry;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.SystemUtils;

import static ru.yandex.direct.common.util.HttpUtil.getRemoteAddress;

@ParametersAreNonnullByDefault
public class LogHelper {
    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final int PARTITION_SIZE = 1000;

    private final LogType logType;
    private final String hostName;

    public LogHelper(@Nullable LogType logType) {
        this.logType = logType;
        hostName = SystemUtils.hostname();
    }

    /**
     * Получить заполненный {@link LogEntry}
     *
     * @param logData данные, записываемые в лог
     * @return заполненный LogEntry
     */
    public <T> LogEntry<T> getLogEntry(T logData) {
        LocalDateTime now = LocalDateTime.now();
        return logEntryWithMetadata(logData, now, null);
    }

    /**
     * Получить стрим заполненных записей лога. logList бьется на чанки по {@value #PARTITION_SIZE} объектов.
     * Для каждого чанка создается запись лога.
     *
     * @param logList список объектов, записываемых в лог
     * @param uid     user id
     * @return Stream заполенных записей лога
     */
    public <T> Stream<LogEntry<List<T>>> getPartitionedEntriesStream(List<T> logList, @Nullable Long uid) {
        LocalDateTime now = LocalDateTime.now();
        return StreamEx.ofSubLists(logList, PARTITION_SIZE)
                .map(logListPart -> logEntryWithMetadata(logListPart, now, uid));
    }

    /**
     * Получить заполненную запись лога
     *
     * @param data данные, записываемых в лог
     * @param now  текущее время
     * @param uid  user id
     * @return запись лога
     */
    private <T> LogEntry<T> logEntryWithMetadata(T data, LocalDateTime now, @Nullable Long uid) {
        return new LogEntry<T>()
                .withUid(uid)
                .withIp(getRemoteAddress().orElse(null))
                .withLogHostname(hostName)
                .withService(Trace.current().getService())
                .withMethod(Trace.current().getMethod())
                .withLogTime(now.format(DATE_TIME_FORMATTER))
                .withReqId(Trace.current().getTraceId())
                .withData(data)
                .withLogType(logType);
    }
}
