package ru.yandex.direct.common.log.service;

import java.util.List;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogEntry;
import ru.yandex.direct.common.log.container.LogType;

import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Сервис логирующий произвольные данные в унифицированном формате data-log.
 * Пример записи:
 * <pre>
 * {@code
 *     {
 *      "log_type": ???,
 *      "data": [
 *          ???,
 *          ???,
 *          ],
 *      "reqid": 5774860888346254116,
 *      "method": "BalanceClient",
 *      "log_hostname": "ppcback01e.yandex.ru",
 *      "log_time": "2017-06-26 00:00:29",
 *      "service": "direct.intapi"
 *     }
 * }
 * </pre>
 */
@Service
public class CommonDataLogService {
    private static final Logger LOGGER = LoggerFactory.getLogger("COMMON_DATA.log");

    private final LogHelper logHelper;

    @Autowired
    public CommonDataLogService() {
        this.logHelper = new LogHelper(LogType.UNKNOWN);
    }

    public void log(LogType logType, List<Object> logData, @Nullable Long uid) {
        logHelper.getPartitionedEntriesStream(logData, uid)
                .forEach(l -> logSingleEntry(logType, l));
    }

    public void log(LogType logType, Object logData) {
        logSingleEntry(logType, logHelper.getLogEntry(logData));
    }

    private void logSingleEntry(LogType logType, LogEntry logEntry) {
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info(toJson(logEntry.withLogType(logType)));
        }
    }
}
