package ru.yandex.direct.common.log.service;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogActiveOrdersData;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Сервис логирующий активные заказы.
 * Пример записи:
 * <pre>
 * {
 * "uid" : 1,
 * "method" : "...",
 * "service" : "direct.web",
 * "ip" : "127.0.0.1",
 * "reqid" : 4303498255735922629,
 * "log_hostname" : "77.88.0.36-spb.dhcp.yndx.net",
 * "log_time" : "2017-05-11 15:18:27",
 * "log_type" : "ppclog_price",
 * "data" : [
 * {"UpdateTime":"2019-08-01 01:59:00",
 * "Cost":"2201.048235",
 * "OrderID":"12269358",
 * "Shows":"2339450",
 * "Clicks":"3416",
 * "CostCur":"66278.771000"}
 * ]
 * }
 * </pre>
 */
@Service
public class LogActiveOrdersService {

    private static final String LOGGER_NAME = "ACTIVE_ORDERS.log";
    private static final Logger ACTIVE_ORDERS_LOGGER = LoggerFactory.getLogger(LOGGER_NAME);

    private final LogHelper logHelper;

    public LogActiveOrdersService() {
        this.logHelper = new LogHelper(LogType.ACTIVE_ORDERS);
    }

    /**
     * Добавить в лог запись об окативных заказых
     *
     * @param logDataList активные заказы
     */
    public void logActiveOrders(List<LogActiveOrdersData> logDataList) {
        logHelper.getPartitionedEntriesStream(logDataList, null)
                .map(JsonUtils::toJson)
                .forEach(ACTIVE_ORDERS_LOGGER::info);
    }
}
