package ru.yandex.direct.common.log.service

import com.fasterxml.jackson.databind.PropertyNamingStrategy.SnakeCaseStrategy
import com.fasterxml.jackson.databind.annotation.JsonNaming
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.common.log.LogHelper
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.utils.compress.GzipUtils
import java.time.LocalDateTime
import java.util.Base64

@JsonNaming(SnakeCaseStrategy::class)
class LogBsExportEssDataChunked<T>(
    val logTime: String,
    val spanId: Long,
    val type: String,
    val data: List<LogBsExportEssData<T>>
)

@Service
open class LogBsExportEssService {
    companion object {
        private const val LOGGER_NAME = "BS_EXPORT_ESS.log"
        private val BS_EXPORT_LOGGER = LoggerFactory.getLogger(LOGGER_NAME)
        private const val CHUNK_SIZE = 50
    }

    fun <T> logData(bsExportDataList: List<LogBsExportEssData<T>>, type: String) {
        if (bsExportDataList.isEmpty()) return

        bsExportDataList.chunked(CHUNK_SIZE)
            .parallelStream() // чтобы сжатие данных не тормозило транспорт
            .map { logDataChunk(it, type) }
            .forEachOrdered {
                // писать хотим из "родного" потока
                BS_EXPORT_LOGGER.info(compressString(it))
            }
    }

    private fun <T> logDataChunk(bsExportDataList: List<LogBsExportEssData<T>>, type: String): String {
        val logTime = LocalDateTime.now().format(LogHelper.DATE_TIME_FORMATTER)
        val spanId = Trace.current().spanId
        val logData = LogBsExportEssDataChunked(
            logTime = logTime,
            spanId = spanId,
            type = type,
            data = bsExportDataList
        )
        return JsonUtils.toJson(logData)
    }

    internal fun compressString(data: String) =
        Base64.getEncoder().encodeToString(
            GzipUtils.compressString(data)
        )
}
