package ru.yandex.direct.common.log.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogCampaignBalanceData;
import ru.yandex.direct.common.log.container.LogEntry;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Сервис логирующий изменение суммы кампании.
 * Пример записи:
 * <pre>
 * {@code
 *     {
 *      "log_type": "campaign_balance",
 *      "data": {
 *          "type": "wallet",
 *          "cid": 16846691,
 *          "currency": "RUB",
 *          "ClientID": "8403967",
 *          "sum": "0",
 *          "tid": "8870576711178",
 *          "sum_delta": 0
 *      },
 *      "reqid": 5774860888346254116,
 *      "method": "BalanceClient",
 *      "log_hostname": "ppcback01e.yandex.ru",
 *      "log_time": "2017-06-26 00:00:29",
 *      "service": "direct.intapi"
 *     }
 * }
 * </pre>
 */
@Service
public class LogCampaignBalanceService {
    private static final String LOGGER_NAME = "CAMPAIGN_BALANCE.log";
    private static final Logger BALANCE_LOGGER = LoggerFactory.getLogger(LOGGER_NAME);

    private final LogHelper logHelper;

    /**
     * Конструктор нужен для тестов для проставления LogHelper
     */
    LogCampaignBalanceService(LogHelper logHelper) {
        this.logHelper = logHelper;
    }

    @Autowired
    public LogCampaignBalanceService() {
        this.logHelper = new LogHelper(LogType.CAMPAIGN_BALANCE);
    }

    /**
     * Добавить в лог запись изменений суммы кампании
     *
     * @param logData параметры изменений суммы кампании
     */
    public void logCampaignBalance(LogCampaignBalanceData logData) {
        LogEntry logEntry = logHelper.getLogEntry(logData);
        BALANCE_LOGGER.info(JsonUtils.toJson(logEntry));
    }
}
