package ru.yandex.direct.common.log.service;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.CharMatcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.common.log.service.metrics.MetricData;
import ru.yandex.direct.common.log.service.metrics.MetricsAddRequest;
import ru.yandex.direct.common.log.service.metrics.MetricsLogHttpSender;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.logging.GlobalCustomMdc;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис, логирующий различные метрики.
 * Может писать данные либо в лог, либо отправлять в intapi-ручку.
 * <p>
 * Пример записи:
 * <pre>
 * {@code
 *     {"log_time":"2019-12-03 18:31:34","method":"metrics.add","service":"direct.intapi","ip":"0:0:0:0:0:0:0:1",
 *     "reqid":0,"log_hostname":"141.8.150.219","log_type":"metrics",
 *     "data":[{"name":"YA_IDEA","value":83.0,"context":{"login":"yandex_login","path":"/direct"}}]}
 * }
 * </pre>
 */

@Service
public class LogMetricsService {
    private static final Logger logger = LoggerFactory.getLogger(LogMetricsService.class);
    private final LogHelper logHelper = new LogHelper(LogType.METRICS);
    private static final Logger METRICS_LOGGER = LoggerFactory.getLogger("METRICS.log");
    private final boolean httpTransportEnabled;
    private final MetricsLogHttpSender metricsLogHttpSender;
    private final HashSet<String> solomonAllowed;
    private final MetricRegistry solomonRegistry;
    private final CharMatcher solomonLabelAntiMatcher = CharMatcher.inRange('a', 'z')
            .or(CharMatcher.inRange('A', 'Z'))
            .or(CharMatcher.inRange('0', '9'))
            .or(CharMatcher.anyOf("_-."))
            .negate();

    @Autowired
    public LogMetricsService(
            @Value("${metrics_log.http_transport_enabled}") boolean httpTransportEnabled,
            DirectConfig directConfig,
            MetricsLogHttpSender metricsLogHttpSender
    ) {
        this.httpTransportEnabled = httpTransportEnabled;
        this.metricsLogHttpSender = metricsLogHttpSender;
        this.solomonAllowed = new HashSet<>(directConfig.getStringList("metrics_log.solomon_allowed_list"));
        this.solomonRegistry = SolomonUtils.SOLOMON_REGISTRY.subRegistry(Labels.of("type", "metrics_log"));
        initSolomonMetrics();
    }

    public CompletableFuture<Boolean> sendMetricAsync(
            String name, double value, Map<String, String> context
    ) {
        var mergedContext = new HashMap<>(Map.of(
                "env", Environment.getCached().toString(),
                "hostname", SystemUtils.hostname(),
                "login", System.getProperty("user.name"),
                "service", nvl(GlobalCustomMdc.getValue(TraceMdcAdapter.SERVICE_KEY), "unknown")
        ));
        mergedContext.putAll(context);
        var metricData = new MetricData(name, value, mergedContext);

        if (httpTransportEnabled) {
            logger.info("async send metric to http: {}={}, context={}", name, value, mergedContext);
            var req = new MetricsAddRequest(new MetricData[]{metricData}, Map.of());
            return metricsLogHttpSender.sendMetrics(req)
                    .thenApply(MetricsLogHttpSender.Response::isSuccessful);
        } else {
            saveMetrics(List.of(metricData));
            return CompletableFuture.completedFuture(true);
        }
    }

    public void saveMetrics(List<MetricData> logMetricsList) {
        // пишем метрики в файл
        logHelper.getPartitionedEntriesStream(logMetricsList, null)
                .map(JsonUtils::toJson)
                .forEach(METRICS_LOGGER::info);
        // избранные метрики пишем в Solomon
        logMetricsList.stream()
                .filter(m -> solomonAllowed.contains(m.getName()))
                .forEach(m ->
                        solomonProcessMetric(m.getName(), 1, m.getValue())
                );
    }

    private void initSolomonMetrics() {
        for (String metric : solomonAllowed) {
            solomonProcessMetric(metric, 0, 0);
        }
    }

    private void solomonProcessMetric(String metric, int count, double value) {
        var sanitizedName = solomonLabelAntiMatcher.replaceFrom(metric, '_');
        solomonRegistry.rate("metrics.count", Labels.of("metric_name", sanitizedName))
                .add(count);
        solomonRegistry.gaugeDouble("metrics.sum", Labels.of("metric_name", sanitizedName))
                .add(value);
    }
}
