package ru.yandex.direct.common.log.service;


import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.LogHelper;
import ru.yandex.direct.common.log.container.LogEntry;
import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.utils.JsonUtils;


/**
 * Сервис логирующий ставки.
 * Пример записи:
 * <pre>
 * {
 * "uid" : 1,
 * "method" : "...",
 * "service" : "direct.web",
 * "ip" : "127.0.0.1",
 * "reqid" : 4303498255735922629,
 * "log_hostname" : "77.88.0.36-spb.dhcp.yndx.net",
 * "log_time" : "2017-05-11 15:18:27",
 * "log_type" : "ppclog_price",
 * "data" : [
 * {
 * "cid" : 0,
 * "pid" : 0,
 * "bid" : 0,
 * "id" : 0,
 * "price_ctx" : 0.0,
 * "price" : 0.0,
 * "type" : "update",
 * "currency" : "YND_FIXED"
 * }
 * ]
 * }
 * </pre>
 */
@Service
public class LogPriceService {
    private static final LogType LOG_TYPE = LogType.PPCLOG_PRICE;
    private static final String LOGGER_NAME = "PPCLOG_PRICE.log";

    private final LogHelper logHelper;

    private final Logger logger = LoggerFactory.getLogger(LOGGER_NAME);

    @Autowired
    public LogPriceService() {
        this.logHelper = new LogHelper(LOG_TYPE);
    }

    public void logPrice(List<LogPriceData> logPriceDataList, long uid) {
        logHelper.getPartitionedEntriesStream(logPriceDataList, uid)
                .map(this::entryToString)
                .forEach(logger::info);
    }

    private String entryToString(LogEntry<?> logEntry) {
        return JsonUtils.toJson(logEntry);
    }
}
