package ru.yandex.direct.common.log.service.metrics;

import java.util.concurrent.CompletableFuture;

import org.asynchttpclient.AsyncHttpClient;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.POST;

/**
 * Отправка метрик в metrics_log через intapi
 * (может пригодиться, при отправке с ноута или бет, где metrics_log автоматически не собирается    )
 */
@Service
public class MetricsLogHttpSender {
    private final AsyncHttpClient asyncHttpClient;
    private final Api api;

    public MetricsLogHttpSender(
            @Value("${metrics_log.url}") String baseUrl,
            AsyncHttpClient asyncHttpClient
    ) {
        this.asyncHttpClient = asyncHttpClient;

        // сам по себе фетчер использоваться не будет, поэтому конфиг
        var parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        this.api = Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("metrics_log:send")
                .withBaseUrl(baseUrl)
                .build()
                .create(Api.class);
    }

    public CompletableFuture<Response> sendMetrics(MetricsAddRequest req) {
        // Хочется асинхронности, поэтому используем smart только для формирования запросов
        var parsableRequest = api.metricsAdd(req).getRequest();
        var ahcRequest = parsableRequest.getAHCRequest();
        return asyncHttpClient.executeRequest(ahcRequest)
                .toCompletableFuture()
                .thenApply(parsableRequest.getParseFunction());
    }

    public interface Api {
        @POST("/metrics/add")
        @Json
        @Headers("Content-Type: application/json")
        Call<Response> metricsAdd(@Body @Json MetricsAddRequest payload);
    }

    public static class Response {
        boolean successful;

        public void setSuccessful(boolean successful) {
            this.successful = successful;
        }

        public boolean isSuccessful() {
            return successful;
        }
    }

}
