package ru.yandex.direct.common.logging;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.SystemUtils;

@SuppressWarnings({"FieldCanBeLocal", "unused", "WeakerAccess"})
public abstract class LogRecord {

    private static final DateTimeFormatter PREFIX_LOG_TIME_FORMATTER =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter JSON_LOG_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    @JsonProperty("logtime")
    private String logTime;

    @JsonProperty("reqid")
    private String id;

    @JsonProperty("trace_id")
    private Long traceId;

    @JsonProperty("ip")
    private String userIp;

    @JsonProperty("host")
    private String host;

    @JsonProperty("cmd")
    private String path;

    @JsonProperty("uid")
    private Long operatorUid;

    @JsonProperty("cids")
    private Collection<Long> cids;

    @JsonProperty("pids")
    private Collection<Long> pids;

    @JsonProperty("bids")
    private Collection<Long> bids;

    @JsonProperty("http_status")
    private long httpStatus;

    @JsonProperty("runtime")
    private double runtime;

    @JsonProperty("cpu_user_time")
    private double cpuUserTime;

    @JsonProperty("response")
    private String response;

    @JsonProperty("tvm_service_id")
    private Integer tvmServiceId;

    private Map<String, Object> allParameters = new HashMap<>();

    @JsonIgnore
    private String prefixLogTime;

    public LogRecord(long requestId, HttpServletRequest request) {
        LocalDateTime currentTime = LocalDateTime.now();
        prefixLogTime = currentTime.format(PREFIX_LOG_TIME_FORMATTER);
        logTime = currentTime.format(JSON_LOG_TIME_FORMATTER);
        id = Long.toString(requestId);
        userIp = HttpUtil.getRemoteAddress(request).getHostAddress();
        host = SystemUtils.hostname();
        path = Trace.current().getMethod();
        traceId = Trace.current().getTraceId();
        saveQueryParameters(request);

    }

    public LogRecord(long requestId) {
        LocalDateTime currentTime = LocalDateTime.now();
        prefixLogTime = currentTime.format(PREFIX_LOG_TIME_FORMATTER);
        logTime = currentTime.format(JSON_LOG_TIME_FORMATTER);
        id = Long.toString(requestId);
        host = SystemUtils.hostname();
        path = Trace.current().getMethod();
        traceId = Trace.current().getTraceId();

    }

    private void saveQueryParameters(HttpServletRequest request) {
        Map<String, String[]> parameterMap = request.getParameterMap();
        for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
            allParameters.put(entry.getKey(), entry.getValue()[0]);
        }
    }

    public void setOperatorId(Long operatorId) {
        this.operatorUid = operatorId;
    }

    public void setPath(String newPath) {
        path = newPath;
    }

    public void setCids(Collection<Long> cids) {
        this.cids = cids;
    }

    public void setPids(Collection<Long> pids) {
        this.pids = pids;
    }

    public void setBids(Collection<Long> bids) {
        this.bids = bids;
    }

    public void setRequestObject(Object object) {
        allParameters.put("_json", object);
    }

    public void setResponse(String response) {
        this.response = response;
    }

    public void setHttpStatus(long httpStatus) {
        this.httpStatus = httpStatus;
    }

    public void setRuntime(double runtime) {
        this.runtime = runtime;
    }

    public void setCpuUserTime(double cpuUserTime) {
        this.cpuUserTime = cpuUserTime;
    }

    public Collection<Long> getCids() {
        return cids;
    }

    public String getPath() {
        return path;
    }

    public Long getOperatorUid() {
        return operatorUid;
    }

    public Integer getTvmServiceId() {
        return tvmServiceId;
    }

    public void setTvmServiceId(Integer tvmServiceId) {
        this.tvmServiceId = tvmServiceId;
    }

    @JsonProperty("param")
    public Map<String, Object> getParam() {
        return allParameters;
    }

    public String getPrefixLogTime() {
        return prefixLogTime;
    }

    @JsonProperty("service")
    public abstract String getService();
}
