package ru.yandex.direct.common.logging;

import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ParametersAreNonnullByDefault
public class LoggingConfigurationUtils {
    private static final Logger logger = LoggerFactory.getLogger(LoggingConfigurationUtils.class);

    private static final String CONFIG_ATTRIBUTE_NAME = LoggingConfigurationUtils.class.getCanonicalName();

    private LoggingConfigurationUtils() {
    }

    /**
     * Поискать настройки логирования в атрибутах запроса
     *
     * @param request запрос из аттрибутов которого попробуем получить настроки
     */
    public static Optional<LoggingConfig> findConfig(HttpServletRequest request) {
        Object obj = request.getAttribute(CONFIG_ATTRIBUTE_NAME);
        if (obj != null && obj instanceof LoggingConfig) {
            return Optional.of((LoggingConfig) obj);
        } else {
            return Optional.empty();
        }
    }

    /**
     * Сохранить настроки логирования
     *
     * @param request запрос, в аттрибутах которго сохраним настройки
     * @param config  настройки для сохранения
     */
    public static void saveConfig(HttpServletRequest request, LoggingConfig config) {
        request.setAttribute(CONFIG_ATTRIBUTE_NAME, config);
    }

    /**
     * Отключить логгирование конкретного запроса
     *
     * @param request
     */
    public static void disableLogging(HttpServletRequest request) {
        try {
            LoggingConfig config = Configs.class
                    .getDeclaredMethod("disabledLoggingConfig")
                    .getAnnotation(LoggingConfig.class);
            LoggingConfigurationUtils.saveConfig(request, config);
        } catch (NoSuchMethodException e) {
            logger.error("Can't find disabledLoggingConfig method");
        }
    }

    static class Configs {
        @LoggingConfig(enabled = EventType.NONE)
        private void disabledLoggingConfig() {
            // nope
        }
    }
}
