package ru.yandex.direct.common.logging;

import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Настройки логгирования запросов/ответов, настраивается для конкретного приложения,
 * перекрываются значениями из аннотации LoggingConfig
 */
@ParametersAreNonnullByDefault
public class LoggingSettings {
    private final EventType enabled;

    private final EventType logRequestBody;
    private final int maxRequestBodySize;

    private final EventType logResponseBody;
    private final int maxResponseBodySize;

    public LoggingSettings(int maxRequestBodySize, int maxResponseBodySize) {
        this(EventType.ALL,
                EventType.ALL, maxRequestBodySize,
                EventType.ERRORS, maxResponseBodySize
        );
    }

    @SuppressWarnings("CheckReturnValue")
    public LoggingSettings(EventType enabled,
                           EventType logRequestBody, int maxRequestBodySize,
                           EventType logResponseBody, int maxResponseBodySize
    ) {
        checkNotNull(logRequestBody);
        checkArgument(enabled != EventType.DEFAULT, "Incorrect enabled: {}", enabled);

        checkArgument(logRequestBody != EventType.DEFAULT, "Incorrect logRequestBody: {}", logRequestBody);
        checkArgument(logRequestBody == EventType.NONE || maxRequestBodySize > 0,
                "Incorrect maxRequestBodySize: {}", maxRequestBodySize);

        checkNotNull(logResponseBody);
        checkArgument(logResponseBody != EventType.DEFAULT, "Incorrect logResponseBody: {}", logResponseBody);
        checkArgument(logResponseBody == EventType.NONE || maxResponseBodySize > 0,
                "Incorrect maxResponseBodySize: {}", maxResponseBodySize);

        this.enabled = enabled;

        this.logRequestBody = logRequestBody;
        this.maxRequestBodySize = maxRequestBodySize;

        this.logResponseBody = logResponseBody;
        this.maxResponseBodySize = maxResponseBodySize;
    }

    /**
     * Создать новый объект настроек, используя текущий объект как дефолт и применить значения, указанные в аннотации
     */
    public LoggingSettings withConfig(LoggingConfig config) {
        return new LoggingSettings(
                config.enabled() != EventType.DEFAULT ? config.enabled() : this.enabled,

                config.logRequestBody() != EventType.DEFAULT ? config.logRequestBody() : this.logRequestBody,
                config.maxRequestBodySize() != LoggingConfig.DEFAULT_BODY_SIZE
                        ? config.maxRequestBodySize() : this.maxRequestBodySize,

                config.logResponseBody() != EventType.DEFAULT ? config.logResponseBody() : this.logResponseBody,
                config.maxResponseBodySize() != LoggingConfig.DEFAULT_BODY_SIZE
                        ? config.maxResponseBodySize() : this.maxResponseBodySize
        );
    }

    /**
     * Включено ли логирование
     */
    public EventType enabled() {
        return enabled;
    }

    /**
     * Для каких результатов выполнения нужно логировать тело запроса
     */
    public EventType logRequestBody() {
        return logRequestBody;
    }

    /**
     * Максимальный размер тела запроса для логгирования
     */
    public int maxRequestBodySize() {
        return maxRequestBodySize;
    }

    /**
     * Для каких результатов выполнения нужно логировать тело ответа
     */
    public EventType logResponseBody() {
        return logResponseBody;
    }

    /**
     * Максимальный размер тела ответа для логгирования
     */
    public int maxResponseBodySize() {
        return maxResponseBodySize;
    }
}
