package ru.yandex.direct.common.logging;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.ServletOutputStream;
import javax.servlet.WriteListener;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.util.StringUtil;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

/**
 * "Проксирующий" {@link HttpServletResponseWrapper}.
 * <p>
 * При включенном логировании методы {@link #getOutputStream()} и {@link #getWriter()} возвращают обертки над
 * {@link ServletOutputStream} и {@link PrintWriter} исходного {@link HttpServletResponse}, которые дополнительно
 * сохраняют данные во внутренний буффер. Сохраняются первые {@code maxBodyBytes} байт ответа, последующие данные
 * будут проксированы без сохранения.
 * <p>
 * При отключенном логировании (умолчание) методы {@link #getOutputStream()} и {@link #getWriter()} возвращают объекты
 * исходного {@code response} без модификации.
 *
 * @see org.springframework.web.util.ContentCachingResponseWrapper
 * @see Response
 */
@ParametersAreNonnullByDefault
public class ProxyResponseWrapper extends HttpServletResponseWrapper {
    private ByteArrayOutputStream content;
    private boolean loggerEnabled;
    private int maxBodyBytes;
    private int capacity;

    private Response.OutputType outputType;
    private ServletOutputStream outputStream;
    private PrintWriter writer;

    /**
     * @param response исходный ответ, в который будем проксировать данные
     */
    public ProxyResponseWrapper(HttpServletResponse response) {
        super(response);
        this.outputType = Response.OutputType.NONE;
    }

    /**
     * Включает проксирование и сохранение ответа
     *
     * @param maxBodyBytes ограничение на размер логируемого ответа
     * @throws IllegalStateException если был вызван после использования {@link #getOutputStream()}
     *                               или {@link #getWriter()}
     */
    public void enableLogger(int maxBodyBytes) {
        checkState(outputType == Response.OutputType.NONE && content == null);
        checkArgument(maxBodyBytes > 0, "maxBodyBytes should be greater than 0");
        this.maxBodyBytes = maxBodyBytes;
        loggerEnabled = true;
    }

    @Nullable
    public String getContentString(Charset charset) {
        if (loggerEnabled && content != null) {
            byte[] buf = content.toByteArray();
            return new String(buf, 0, buf.length < maxBodyBytes ? buf.length : maxBodyBytes, charset);
        } else {
            return null;
        }
    }

    private void initContent() {
        if (content == null) {
            content = new ByteArrayOutputStream();
            updateCapacity();
        }
    }

    private void resetContent() {
        if (content != null) {
            content.reset();
            updateCapacity();
        }
    }

    private void updateCapacity() {
        capacity = maxBodyBytes - content.size();
    }

    private ServletOutputStream getWrappedOutputStream() throws IOException {
        initContent();

        return new ResponseServletOutputStream(super.getOutputStream());
    }

    private PrintWriter getWrappedWriter() throws IOException {
        initContent();

        String characterEncoding = getCharacterEncoding();
        if (characterEncoding == null) {
            // from org.eclipse.jetty.server.Response
            characterEncoding = StringUtil.__ISO_8859_1;
        }

        return new ResponsePrintWriter(super.getWriter(), characterEncoding);
    }

    @Override
    public void reset() {
        super.reset();
        resetContent();
        outputType = Response.OutputType.NONE;
    }

    @Override
    public void resetBuffer() {
        super.resetBuffer();
        resetContent();
    }

    @Override
    public ServletOutputStream getOutputStream() throws IOException {
        if (outputType == Response.OutputType.WRITER) {
            throw new IllegalStateException("WRITER");
        }

        if (outputType == Response.OutputType.NONE) {
            if (loggerEnabled) {
                outputStream = getWrappedOutputStream();
            } else {
                outputStream = super.getOutputStream();
            }
            outputType = Response.OutputType.STREAM;
        }

        return outputStream;
    }

    @Override
    public PrintWriter getWriter() throws IOException {
        if (outputType == Response.OutputType.STREAM) {
            throw new IllegalStateException("STREAM");
        }

        if (outputType == Response.OutputType.NONE) {
            if (loggerEnabled) {
                writer = getWrappedWriter();
            } else {
                writer = super.getWriter();
            }
            outputType = Response.OutputType.WRITER;
        }

        return writer;
    }

    private class ResponsePrintWriter extends PrintWriter {

        private PrintWriter pw;

        ResponsePrintWriter(PrintWriter pw, String characterEncoding) throws UnsupportedEncodingException {
            super(new OutputStreamWriter(content, characterEncoding));
            this.pw = pw;
        }

        @Override
        public void write(char[] buf, int off, int len) {
            this.pw.write(buf, off, len);
            if (capacity > 0) {
                super.write(buf, off, len < capacity ? len : capacity);
                flushInternal();
            }
        }

        @Override
        public void write(String s, int off, int len) {
            this.pw.write(s, off, len);
            if (capacity > 0) {
                super.write(s, off, len < capacity ? len : capacity);
                flushInternal();
            }
        }

        @Override
        public void write(int c) {
            this.pw.write(c);
            if (capacity > 0) {
                super.write(c);
                flushInternal();
            }
        }

        @Override
        public void flush() {
            this.pw.flush();
            flushInternal();
        }


        private void flushInternal() {
            super.flush();
            updateCapacity();
        }
    }

    private class ResponseServletOutputStream extends ServletOutputStream {

        private final ServletOutputStream os;

        ResponseServletOutputStream(ServletOutputStream os) {
            this.os = os;
        }

        @Override
        public void write(int b) throws IOException {
            this.os.write(b);
            if (capacity > 0) {
                content.write(b);
                updateCapacity();
            }
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.os.write(b, off, len);
            if (capacity > 0) {
                content.write(b, off, len <= capacity ? len : capacity);
                updateCapacity();
            }
        }

        @Override
        public boolean isReady() {
            return this.os.isReady();
        }

        @Override
        public void setWriteListener(WriteListener writeListener) {
            this.os.setWriteListener(writeListener);
        }
    }
}
