package ru.yandex.direct.common.mds;

import java.time.Duration;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceEvent;
import ru.yandex.direct.liveresource.LiveResourceListener;
import ru.yandex.direct.liveresource.PollingLiveResourceWatcher;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.solomon.SolomonResponseMonitorStatus;
import ru.yandex.inside.mds.Mds;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsFileNotFoundException;
import ru.yandex.inside.mds.MdsHosts;
import ru.yandex.inside.mds.MdsInternalStaticProxies;
import ru.yandex.inside.mds.MdsMetrics;
import ru.yandex.inside.mds.MdsNamespace;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mds.MdsStorageException;
import ru.yandex.inside.mds.MdsStorageHttpException;
import ru.yandex.inside.mds.MdsStorageKeyAlreadyExistsHttpException;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.ip.HostPort;

import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNKNOWN;

@ParametersAreNonnullByDefault
public class MdsHolder implements LiveResourceListener {
    private static final String EXTERNAL_SYSTEM = "mds";
    private static final String METHOD_UPLOAD = "upload";
    private static final String METHOD_DELETE = "delete";
    private static final String METHOD_DOWNLOAD = "download";
    private static final SolomonExternalSystemMonitorService MONITOR_SERVICE = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(METHOD_UPLOAD, METHOD_DELETE, METHOD_DOWNLOAD)
    );

    private final AtomicReference<Mds> mds = new AtomicReference<>();

    private final String readHostPort;
    private final String writeHostPort;
    private final String namespace;
    private final int timeoutInSeconds;
    private final int maxConnections;

    protected MdsHolder(String readHostPort, String writeHostPort, String namespace, int timeoutInSeconds,
                      int maxConnections) {
        this.readHostPort = readHostPort;
        this.writeHostPort = writeHostPort;
        this.namespace = namespace;
        this.timeoutInSeconds = timeoutInSeconds;
        this.maxConnections = maxConnections;
    }

    @SuppressWarnings("checkstyle:parameternumber")
    public static MdsHolder instance(String readHostPort, String writeHostPort, String namespace, int timeoutInSeconds,
                                     LiveResource tokenResource, TaskScheduler taskScheduler, long checkRate,
                                     int maxConnections) {
        MdsHolder instance = new MdsHolder(readHostPort, writeHostPort, namespace, timeoutInSeconds, maxConnections);
        String initialToken = tokenResource.getContent();
        instance.mds.set(instance.constructMds(initialToken));

        PollingLiveResourceWatcher resourceWatcher =
                new PollingLiveResourceWatcher(tokenResource, initialToken, taskScheduler, checkRate);

        resourceWatcher.addListener(instance);
        resourceWatcher.watch();

        return instance;
    }

    @Override
    public void update(LiveResourceEvent event) {
        String token = event.getCurrentContent();
        mds.set(constructMds(token));
    }

    private Mds constructMds(String token) {
        MdsHosts proxies = new MdsHosts(
                HostPort.parse(readHostPort),
                HostPort.parse(writeHostPort),
                new MdsInternalStaticProxies(HostPort.parse(writeHostPort)));

        MdsNamespace mdsNamespace = new MdsNamespace(namespace, "Basic " + token.trim());

        Duration timeout = Duration.ofSeconds(timeoutInSeconds);
        Timeout mdsTimeout = new Timeout(timeout.toMillis(), timeout.toMillis());

        return new Mds(proxies, mdsNamespace, maxConnections, mdsTimeout, MdsMetrics.EMPTY);
    }

    private Mds getMds() {
        return mds.get();
    }

    public MdsHosts getHosts() {
        return getMds().getHosts();
    }

    public MdsNamespace getNamespace() {
        return getMds().getNamespace();
    }

    public String downloadUrl(String action, MdsFileKey fileKey) {
        return getMds().downloadUrl(action, fileKey);
    }

    public MdsPostResponse upload(String filename, InputStreamSource source) {
        try {
            MdsPostResponse upload = getMds().upload(filename, source);
            MONITOR_SERVICE.write(METHOD_UPLOAD, STATUS_2XX);
            return upload;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_UPLOAD, monitorStatus);
            throw e;
        }
    }

    public MdsPostResponse upload(String filename, InputStreamSource source, org.joda.time.Duration expire) {
        try {
            MdsPostResponse upload = getMds().upload(filename, source, expire);
            MONITOR_SERVICE.write(METHOD_UPLOAD, STATUS_2XX);
            return upload;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_UPLOAD, monitorStatus);
            throw e;
        }
    }

    public void delete(MdsFileKey mdsFileKey) {
        try {
            getMds().delete(mdsFileKey);
            MONITOR_SERVICE.write(METHOD_DELETE, STATUS_2XX);
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_DELETE, monitorStatus);
            throw e;
        }
    }

    public InputStreamSource download(MdsFileKey mdsFileKey) {
        try {
            var source = getMds().download(mdsFileKey);
            MONITOR_SERVICE.write(METHOD_DOWNLOAD, STATUS_2XX);
            return source;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_DOWNLOAD, monitorStatus);
            throw e;
        }
    }

    private static SolomonResponseMonitorStatus guessStatusByException(RuntimeException exception) {
        if (exception instanceof MdsStorageKeyAlreadyExistsHttpException
                || exception instanceof MdsFileNotFoundException) {
            return STATUS_4XX;
        }
        if (exception instanceof MdsStorageHttpException) {
            try {
                int code = ((MdsStorageHttpException) exception).getStatusLine().get().getStatusCode() / 100;
                if (code == 4) {
                    return STATUS_4XX;
                }
            } catch (RuntimeException ignored) {
            }
            return STATUS_5XX;
        }
        if (exception instanceof MdsStorageException) {
            return STATUS_5XX;
        }
        return STATUS_UNKNOWN;
    }
}
