package ru.yandex.direct.common.metrics;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.util.concurrent.AtomicDouble;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Плагин для metric-collector, предоставляет статистику о обработанных запросах в разбивке по кодам ответов.
 * Статистика пишется по пути: reqs.ControllerClassName.methodName.*
 * Метрики пишутся с накоплением, чтобы просыпанные данные не терялись.
 * <p>
 * Список метрик:
 * <ul>
 * <li>time_ms - время выполнения запросов, в миллисекундах</li>
 * <li>requests - количество запросов</li>
 * <li>responses_2xx - количество ответов со статус-кодами 2хх</li>
 * <li>responses_3xx</li>
 * <li>responses_4xx</li>
 * <li>responses_5xx</li>
 * <li>responses_unknown - количество ответов с прочими статус-кодами</li>
 * </ul>
 * <p>
 * Для корректного определения имени контроллера и метода - требуется {@link MetricsInterceptor} в цепочке интерсепторов
 *
 * @see MetricsInterceptor
 */
@Component
@ParametersAreNonnullByDefault
public class MetricsFilter extends OncePerRequestFilter {
    static final String METHOD_ATTRIBUTE_NAME = MetricsFilter.class.getCanonicalName();
    static final Pattern ALLOWED_METHOD_NAME_SYMBOLS = Pattern.compile("[^a-zA-Z0-9_\\-]");
    static final Pattern MULTI_UNDERSCORE = Pattern.compile("_+");

    private static final String METRIC_PREFIX = "reqs";
    private final Map<String, AtomicDouble> stats = new ConcurrentHashMap<>();

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        boolean isFirstRequest = !isAsyncDispatch(request);
        long reqStartNanotime = System.nanoTime();

        try {
            filterChain.doFilter(request, response);
        } finally {
            if (isFirstRequest && !isAsyncStarted(request)) {
                addMetrics(request, response, reqStartNanotime);
            }
        }
    }

    @SuppressWarnings("CheckReturnValue")
    private void addMetrics(HttpServletRequest request, HttpServletResponse response, long reqStartNanotime) {
        Method method = (Method) request.getAttribute(MetricsFilter.METHOD_ATTRIBUTE_NAME);

        String controller;
        String methodStr;
        if (method == null) {
            controller = "path";
            if (Trace.current().getMethod().isEmpty()) {
                methodStr = "unknown";
            } else {
                methodStr = toCleanMethodName(Trace.current().getMethod().replace('.', '_'));
            }
        } else {
            controller = method.getDeclaringClass().getSimpleName();
            methodStr = method.getName();
        }

        String path = controller + "." + methodStr;

        int statusHigh = response.getStatus() / 100;
        String status = statusHigh >= 2 && statusHigh <= 5
                ? statusHigh + "xx"
                : "unknown";

        double elaMs = ((double) (System.nanoTime() - reqStartNanotime)) / TimeUnit.MILLISECONDS.toNanos(1);
        stats.computeIfAbsent(path + ".time_ms", x -> new AtomicDouble(0))
                .addAndGet(elaMs);
        stats.computeIfAbsent(path + ".requests", x -> new AtomicDouble(0))
                .addAndGet(1);
        stats.computeIfAbsent(path + ".responses_" + status, x -> new AtomicDouble(0))
                .addAndGet(1);

        Labels labels = Labels.of("controller", controller, "method", methodStr, "status", status);
        SOLOMON_REGISTRY.counter(METRIC_PREFIX + ".time_ms", labels)
                .add((long) elaMs);
        SOLOMON_REGISTRY.counter(METRIC_PREFIX + ".count", labels)
                .add(1);
    }

    static String toCleanMethodName(String methodName) {
        return MULTI_UNDERSCORE.matcher(
                ALLOWED_METHOD_NAME_SYMBOLS.matcher(methodName).replaceAll("_")
        ).replaceAll("_");
    }

    @Override
    protected boolean shouldNotFilterAsyncDispatch() {
        return false;
    }
}
