package ru.yandex.direct.common.net;

import java.math.BigInteger;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;

import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import com.google.common.net.InetAddresses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Хранит набор IP-сетей и позволяет быстро проверить, попадает ли IP-адрес в набор.
 */
public class IpRangeSetValidator {
    private static final Logger logger = LoggerFactory.getLogger(IpRangeSetValidator.class);

    //IPv6 и IPv4-адреса хранятся в общем дереве(IPv4 преобразуются в IPv6 - т.н. IPv4-to-IPv6 ).
    //Другой вариант - хранить в префиксном дереве.
    //Обычно применяется PATRICIA trie (в т.ч. его использует Perl - версия Direct Api), но версию PatriciaTrie из
    //Apache Commons Collections(org.apache.commons.collections4.trie.PatriciaTrie) сложнее настраивать.
    private final RangeSet<BigInteger> rangeSet;

    // для Host Based Firewall нужны отдельные проверки на каждую сеть
    private final Collection<ProjectNetRange> projectNetRanges;

    public IpRangeSetValidator() {
        this.rangeSet = TreeRangeSet.create();
        this.projectNetRanges = new ArrayList<>();
    }

    public static IpRangeSetValidator fromMasksSafely(Collection<String> masks) {
        IpRangeSetValidator validator = new IpRangeSetValidator();
        for (String mask : masks) {
            try {
                validator.addMask(mask);
            } catch (RuntimeException ex) {
                logger.error("Incorrect network: " + mask, ex);
            }
        }
        return validator;
    }

    public void addMask(String mask) {
        if (mask.indexOf(ProjectNetRange.PROJ_SEPARATOR) >= 0) {
            projectNetRanges.add(ProjectNetRange.parse(mask));
        } else {
            rangeSet.add(NetRangeParser.parseSingleNetwork(mask));
        }
    }

    public boolean contains(String address) {
        return contains(InetAddresses.forString(address));
    }

    /**
     * @param address проверяемый адрес
     * @return входит ли адрес в набор
     */
    public boolean contains(InetAddress address) {
        BigInteger ip = IpUtils.address2BigInteger(address);
        return rangeSet.contains(ip) || projectNetRanges.stream().anyMatch(r -> r.contains(ip));
    }
}
