package ru.yandex.direct.common.net;

import java.math.BigInteger;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;

import com.google.common.base.Splitter;
import com.google.common.net.InetAddresses;

import static com.google.common.base.Preconditions.checkArgument;

//частично заменим кодом из Iceberg - если доделают трансляцию IPv4-to-Ipv6.
public class IpUtils {

    private static final int IPV4_PART_COUNT = 4;
    private static final Splitter IPV4_SPLITTER = Splitter.on('.').limit(IPV4_PART_COUNT);

    private IpUtils() {
    }

    //при создании InetAddress тип исходного адреса может потеряться
    //(в случае, если это IPv4-to-IPv6 адрес - он будет преобразован в Inet4Address).
    // В случае работы с CIDR-масками это критично.

    /**
     * пытается угадать тип адреса по его строковому представлению.
     *
     * @param address проверяемый адрес
     * @return тип адреса
     */
    public static IpFamilyType guessIpFamily(String address) {
        return address.contains(":") ? IpFamilyType.IPV6 : IpFamilyType.IPV4;
    }

    public static IpFamilyType ipFamily(InetAddress address) {
        return address instanceof Inet6Address ? IpFamilyType.IPV6 : IpFamilyType.IPV4;
    }

    /**
     * преобразует адрес в IPv6-число
     *
     * @param address преобразуемый адрес
     * @return целочисленное представление адреса
     */
    public static BigInteger address2BigInteger(String address) {
        return address2BigInteger(InetAddresses.forString(address));
    }

    /**
     * преобразует адрес в IPv6-число
     *
     * @param address преобразуемый адрес
     * @return целочисленное представление адреса
     */
    public static BigInteger address2BigInteger(InetAddress address) {
        byte[] addressBytes = address.getAddress();
        BigInteger converted = new BigInteger(1, addressBytes);
        converted = IpUtils.ipFamily(address) == IpFamilyType.IPV6 ? converted : ipv4ToIpv6(converted);
        return converted;
    }

    /**
     * преобразует целочисленное представление из IPv4 в IPv6
     *
     * @param ipBits преобразовываемый адрес, ipv4
     * @return преобразованный адрес, ipv6
     */
    public static BigInteger ipv4ToIpv6(BigInteger ipBits) {
        checkArgument(IpFamilyType.IPV4.getFamilyOnes().compareTo(ipBits) >= 0);
        return ipBits.or(IpConstants.IPV4_TO_IPV6_TRANSLATION_PREFIX);
    }

    public static boolean isPrivateAddress(InetAddress inetAddress) {
        // AnyLocalAddress     0.0.0.0
        // MulticastAddress    224.0.0.0/4
        // LoopbackAddress     127.0.0.0/8
        // LinkLocalAddress    169.254.0.0/16
        //
        // SiteLocalAddress refer to RFC 1918:
        // 10/8
        // 172.16/12
        // 192.168/16
        return inetAddress.isAnyLocalAddress() ||   // 0.0.0.0
                inetAddress.isMulticastAddress() || // 224.0.0.0/4
                inetAddress.isLoopbackAddress() ||  // 127.0.0.0/8
                inetAddress.isLinkLocalAddress() || // 169.254.0.0/16
                inetAddress.isSiteLocalAddress();
    }

    /**
     * Поддерживает только ipv4 адреса
     * <p>
     * Аналогично валидации в perl Direct::Validation::Ips::validate_disabled_ips
     * Требуется своя реализация, отличающаяся от java, т.к. в БД уже есть провалидированные значения
     * <p>
     * Отличается тем, что допускает группы из нескольких нулей: 165.00.000.0
     * <p>
     * По мотивам {@link InetAddresses#isInetAddress(java.lang.String)}
     */
    public static boolean isValidIp(String ipString) {
        return ipStringToBytes(ipString) != null;
    }

    /**
     * В отличии от {@link InetAddresses#forString(java.lang.String)} толерантно к [] вокруг адреса
     */
    public static InetAddress ipFromStringWithOptionalBrackets(String ipString) {
        if (ipString.startsWith("[") && ipString.endsWith("]")) {
            ipString = ipString.substring(1, ipString.length() - 1);
        }
        return InetAddresses.forString(ipString);
    }

    /**
     * Поддерживает только ipv4 адреса
     * <p>
     * Допускает группы из нескольких нулей: 165.00.000.0
     * <p>
     * По мотивам {@link InetAddresses#forString(java.lang.String)}
     */
    public static InetAddress ipFromString(String ipString) {
        byte[] bytes = ipStringToBytes(ipString);
        if (bytes == null) {
            throw new IllegalArgumentException(String.format("'%s' is not an IP string literal.", ipString));
        }

        try {
            return InetAddress.getByAddress(bytes);
        } catch (UnknownHostException e) {
            // UnknownHostException невозможен, IP уже проверен на валидность
            throw new AssertionError(e);
        }
    }

    /**
     * Измененная реализация из {@link InetAddresses#forString(java.lang.String)}
     * <p>
     * Отличается тем, что допускает группы из нескольких нулей: 165.00.000.0
     */
    private static byte[] ipStringToBytes(String ipString) {
        for (int i = 0; i < ipString.length(); i++) {
            char c = ipString.charAt(i);
            if (c != '.' && Character.digit(c, 10) == -1) {
                return null;
            }
        }

        byte[] bytes = new byte[IPV4_PART_COUNT];
        int i = 0;
        try {
            for (String octet : IPV4_SPLITTER.split(ipString)) {
                bytes[i++] = parseOctet(octet);
            }
        } catch (NumberFormatException ex) {
            return null;
        }

        return i == IPV4_PART_COUNT ? bytes : null;
    }

    private static byte parseOctet(String ipPart) {
        int octet = Integer.parseInt(ipPart);
        if (octet > 255 || (ipPart.startsWith("0") && octet > 0) || ipPart.length() > 3) {
            throw new NumberFormatException();
        }
        return (byte) octet;
    }
}
