package ru.yandex.direct.common.net;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceEvent;
import ru.yandex.direct.liveresource.LiveResourceListener;
import ru.yandex.direct.liveresource.LiveResourceWatcher;
import ru.yandex.direct.liveresource.LiveResourceWatcherFactory;

/**
 * Инкапсулирует список известных сетей. Умеет:
 * 1) Загружать список сетей из json config-файла
 * 2) Проверять ip-адреса на вхождение в известные сети.
 */
public class NetAcl implements LiveResourceListener {
    private static final String INTERNAL_NETWORKS_SECTION = "internal";
    private static final Logger logger = LoggerFactory.getLogger(NetAcl.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private volatile Map<String, IpRangeSetValidator> networkDefinitions;

    public NetAcl(String initialJson) {
        networkDefinitions = parseNetworksJson(initialJson);
    }


    public static NetAcl createAndWatch(LiveResource liveResource,
                                        LiveResourceWatcherFactory liveResourceWatcherFactory) {
        Assert.notNull(liveResource, "liveResource is required");
        LiveResourceWatcher resourceWatcher = liveResourceWatcherFactory.createWatcher(liveResource);
        NetAcl result = new NetAcl(resourceWatcher.watch());
        resourceWatcher.addListener(result);
        return result;
    }

    @Override
    public void update(LiveResourceEvent event) {
        networkDefinitions = parseNetworksJson(event.getCurrentContent());
    }

    /**
     * Проверяет, входит ли адрес во внутренние сети Yandex.
     *
     * @param address проверяемый адрес
     * @return true, если входит, иначе false
     */
    public boolean isInternalIp(InetAddress address) {
        return isIpInNetwork(address, INTERNAL_NETWORKS_SECTION);
    }

    /**
     * Проверяет, входит ли адрес в указанную сеть.
     *
     * @param address     проверяемый адрес
     * @param networkName имя сети
     * @return true, если входит, иначе false
     */
    public boolean isIpInNetwork(InetAddress address, String networkName) {
        IpRangeSetValidator validator = getIpRangeValidator(networkName);
        if (validator != null) {
            return validator.contains(address);
        } else {
            logger.warn("unknown network: '{}'", networkName);
            return false;
        }
    }

    /**
     * @return {@link IpRangeSetValidator} для указанной сети.
     * Если для сети {@code networkName} не задан валидатор, возвращается {@code null}.
     */
    @Nullable
    IpRangeSetValidator getIpRangeValidator(String networkName) {
        return networkDefinitions.get(networkName);
    }

    /**
     * Проверяет, входит ли указанные сети
     *
     * @param address       проверяемый адрес
     * @param networksNames имена сетей
     * @return true, если входит, иначе false
     */
    public boolean isIpInNetworks(InetAddress address, Collection<String> networksNames) {
        for (String networkName : networksNames) {
            if (isIpInNetwork(address, networkName)) {
                return true;
            }
        }
        return false;
    }

    private Map<String, IpRangeSetValidator> parseNetworksJson(String jsonBody) {
        TypeReference<List<NetAclEntry>> typeRef = new TypeReference<List<NetAclEntry>>() {
        };
        try {
            List<NetAclEntry> configObject = OBJECT_MAPPER.readValue(jsonBody, typeRef);
            Map<String, IpRangeSetValidator> result = new HashMap<>();
            for (NetAclEntry aclEntry : configObject) {
                Collection<String> netDefinitions = aclEntry.getMasks()
                        .stream()
                        .map(NetmaskEntry::getMask)
                        .collect(Collectors.toList());
                result.put(aclEntry.getName(), IpRangeSetValidator.fromMasksSafely(netDefinitions));
            }
            return result;
        } catch (IOException ex) {
            throw new NetworkConfigException("Can't parse network-config: " + jsonBody, ex);
        }
    }

    private static class NetAclEntry {
        private final String name;
        private final List<NetmaskEntry> masks;

        @JsonCreator
        NetAclEntry(@JsonProperty("name") String name, @JsonProperty("masks") List<NetmaskEntry> masks) {
            this.name = name;
            this.masks = masks;
        }

        public String getName() {
            return name;
        }

        public List<NetmaskEntry> getMasks() {
            return masks;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class NetmaskEntry {
        private final String mask;

        @JsonCreator
        NetmaskEntry(@JsonProperty("mask") String mask) {
            this.mask = mask;
        }

        public String getMask() {
            return mask;
        }
    }
}

