package ru.yandex.direct.common.net;

import java.math.BigInteger;
import java.net.InetAddress;

import com.google.common.collect.Range;
import com.google.common.net.InetAddresses;
import org.apache.commons.lang3.StringUtils;

public class NetRangeParser {
    private NetRangeParser() {
    }

    /**
     * Строит IP-диапазон(Range&lt;BigInteger&gt;) на основе его строкового представления. Диапазон м.б. представлен
     * в виде: <br/>
     * <ol>
     * <li>StartAddress-EndAddress</li>
     * <li>CIDR (Address/mask)</li>
     * <ul>
     * <li>Если ipv4 маска больше 32 (например, 36), то используется /32</li>
     * </ul>
     * <li>SingleAddress</li>
     * </ol>
     *
     * @param netString строковое представление IP-диапазона
     * @return IP-диапазон
     */
    public static Range<BigInteger> parseSingleNetwork(String netString) {
        final String sanitized = StringUtils.deleteWhitespace(netString);
        if (sanitized.contains("/")) {
            return parseCidrString(sanitized);
        } else {
            if (sanitized.contains("-")) {
                return parseRangeString(sanitized);
            } else {
                return Range.singleton(IpUtils.address2BigInteger(sanitized));
            }
        }
    }

    private static Range<BigInteger> parseRangeString(String maskString) {
        String[] tokens = maskString.split("-", 2);
        IpFamilyType startAddrFamily = IpUtils.guessIpFamily(tokens[0]);
        IpFamilyType endAddrFamily = IpUtils.guessIpFamily(tokens[1]);
        if (startAddrFamily == endAddrFamily) {
            BigInteger startAddr = IpUtils.address2BigInteger(tokens[0]);
            BigInteger endAddr = IpUtils.address2BigInteger(tokens[1]);
            return Range.closed(startAddr, endAddr);
        } else {
            throw new IllegalArgumentException("hybrid range '" + maskString + "' is not allowed");
        }
    }

    private static Range<BigInteger> parseCidrString(String maskString) {
        String[] ipvTokens = maskString.split("/", 2);
        InetAddress maskBodyAddress = InetAddresses.forString(ipvTokens[0]);
        IpFamilyType ipFamilyType = IpUtils.ipFamily(maskBodyAddress);
        int rangeMask = Integer.parseInt(ipvTokens[1]);
        if (ipFamilyType == IpFamilyType.IPV4 && rangeMask > 32) {
            rangeMask = 32;
        }
        int variableBits = ipFamilyType.getBitsCount() - rangeMask;
        BigInteger mask = IpUtils.address2BigInteger(maskBodyAddress);
        return netmaskToMultiaddressRange(mask, IpConstants.IPV6_ADDRESS_BITS_COUNT - variableBits);
    }

    private static Range<BigInteger> netmaskToMultiaddressRange(BigInteger maskBase, int maskBits) {
        int varBitsCount = IpConstants.IPV6_ADDRESS_BITS_COUNT - maskBits;
        BigInteger rightOnes = BigInteger.ONE.shiftLeft(varBitsCount).subtract(BigInteger.ONE);

        BigInteger rightZeros = IpConstants.IPV6_ONES.subtract(rightOnes);
        BigInteger maskStart = maskBase.and(rightZeros);
        BigInteger maskEnd = maskStart.or(rightOnes);
        return Range.closed(maskStart, maskEnd);
    }
}
