package ru.yandex.direct.common.net;

import java.math.BigInteger;
import java.net.InetAddress;
import java.util.List;

import com.google.common.base.Splitter;
import com.google.common.collect.Range;
import com.google.common.net.InetAddresses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetRangeUtil {
    public static final char NETWORK_DELIMITER = ',';
    private static final Splitter TOKENS_SPLITTER = Splitter.on(NETWORK_DELIMITER).trimResults().omitEmptyStrings();
    private static final Logger logger = LoggerFactory.getLogger(NetRangeParser.class);

    private NetRangeUtil() {
    }

    public static boolean isIpInNetworks(String ip, String netsPattern) {
        return NetRangeUtil.isIpInNetworks(InetAddresses.forString(ip), netsPattern);
    }

    /**
     * проверяет, входит ли ip-адрес в набор ip-диапазонов. Диапазоны разделяются запятыми
     *
     * @param ip          проверяемый ip
     * @param netsPattern набор ip-диапазонов
     * @return false если диапазон пуст или ip-адрес не входит в набор.
     */
    public static boolean isIpInNetworks(InetAddress ip, String netsPattern) {
        List<String> tokens = TOKENS_SPLITTER.splitToList(netsPattern);
        if (!tokens.isEmpty()) {
            BigInteger ipv6BigInt = IpUtils.address2BigInteger(ip);
            for (String netToken : tokens) {
                try {
                    Range<BigInteger> parsedRange = NetRangeParser.parseSingleNetwork(netToken);
                    if (parsedRange.contains(ipv6BigInt)) {
                        return true;
                    }
                } catch (IllegalArgumentException ex) {
                    // данные в БД должны быть валидны, поэтому большого количества логов не ожидается
                    logger.warn("Incorrect network '{}'", netToken, ex);
                }
            }
        }
        return false;
    }
}
