package ru.yandex.direct.common.net;

import java.math.BigInteger;
import java.util.Objects;

import com.google.common.collect.Range;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Сеть в понимании host-based-firewall - ipv6-сеть + projectId
 */
public class ProjectNetRange {
    public static final char PROJ_SEPARATOR = '@';

    private static final long MAX_PROJ_ID = (1L << 32) - 1;
    private static final BigInteger BIT_MASK_32 = BigInteger.valueOf(0xFFFF_FFFFL);

    private final long projectId;
    private final Range<BigInteger> ipRange;

    @SuppressWarnings("CheckReturnValue")
    public ProjectNetRange(long projectId, Range<BigInteger> ipRange) {
        checkArgument(projectId > 0, "ProjectId must be positive");
        checkArgument(projectId <= MAX_PROJ_ID, "ProjectId less or equals {}", MAX_PROJ_ID);
        checkNotNull(ipRange);

        this.projectId = projectId;
        this.ipRange = ipRange;
    }

    /**
     * Распарсть строчку с hbf-сетью формата $hex_proj_id@$ipv6_prefix/$mask
     */
    @SuppressWarnings("CheckReturnValue")
    public static ProjectNetRange parse(String str) {
        checkNotNull(str);
        int atPos = str.indexOf(PROJ_SEPARATOR);
        checkArgument(atPos > 0, "String does not contain '@'");

        long projectId = Long.parseLong(str.substring(0, atPos), 16);
        Range<BigInteger> range = NetRangeParser.parseSingleNetwork(str.substring(atPos + 1));

        return new ProjectNetRange(projectId, range);
    }

    /**
     * Содержит ли сеть конкретный адрес (с учётом projId)
     */
    public boolean contains(BigInteger ip) {
        return ipRange.contains(ip) && extractProjectId(ip) == projectId;
    }

    // устройство hbf адресов https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/
    static long extractProjectId(BigInteger ip) {
        return ip.shiftRight(32).and(BIT_MASK_32).longValueExact();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProjectNetRange that = (ProjectNetRange) o;
        return projectId == that.projectId &&
                Objects.equals(ipRange, that.ipRange);
    }

    @Override
    public int hashCode() {
        return Objects.hash(projectId, ipRange);
    }
}
