package ru.yandex.direct.common.spring;

import java.lang.reflect.Field;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.type.StandardMethodMetadata;

/**
 * Помечаем все бины как "ленивые" - инициализируемые только по необходимости.
 * Это нужно для ускорения запуска приложения в разработческих средах.
 */
@BetaComponent
public class LazyInitBeanFactoryPostProcessor implements BeanFactoryPostProcessor {
    private static final Logger logger = LoggerFactory.getLogger(LazyInitBeanFactoryPostProcessor.class);
    public static final String LAZY_ANNOTATION_CANONICAL_NAME = Lazy.class.getCanonicalName();
    /**
     * requestMappingHandlerAdapter - чтобы работала генерация json моделей в интерфейсе swagger
     */
    private static final Set<String> EXTERNAL_EXCEPTIONAL_BEAN_NAMES = Set.of("requestMappingHandlerAdapter");

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        for (String beanName : beanFactory.getBeanDefinitionNames()) {
            BeanDefinition beanDefinition = beanFactory.getBeanDefinition(beanName);
            if (needForceLazyInit(beanName, beanDefinition)) {
                beanDefinition.setLazyInit(true);
            }
        }
    }

    private boolean needForceLazyInit(String beanName, BeanDefinition beanDefinition) {
        return !hasLazyAnnotation(beanDefinition)
                && isNotExceptionalBean(beanName);
    }

    /**
     * Не входит ли бин в список внешних бинов, которые не должны быть Lazy
     */
    private boolean isNotExceptionalBean(String beanName) {
        return !EXTERNAL_EXCEPTIONAL_BEAN_NAMES.contains(beanName);
    }

    /**
     * Если factory method явно проаннотирован Lazy-аннотацией, не изменяем его поведение
     */
    private boolean hasLazyAnnotation(BeanDefinition beanDefinition) {
        if (beanDefinition instanceof AnnotatedBeanDefinition) {
            var metadata = ((AnnotatedBeanDefinition) beanDefinition).getMetadata();
            if (metadata.hasAnnotation(LAZY_ANNOTATION_CANONICAL_NAME)) {
                return true;
            }
        }
        if (beanDefinition.getClass().getCanonicalName().equals("org.springframework.context.annotation" +
                ".ConfigurationClassBeanDefinitionReader.ConfigurationClassBeanDefinition")) {
            try {
                Field factoryMethodMetadata = beanDefinition.getClass().getDeclaredField("factoryMethodMetadata");
                factoryMethodMetadata.setAccessible(true);
                //
                Object methodMetadataObj = factoryMethodMetadata.get(beanDefinition);
                if (!(methodMetadataObj instanceof StandardMethodMetadata)) {
                    return false;
                }
                StandardMethodMetadata methodMetadata = (StandardMethodMetadata) methodMetadataObj;
                //
                Lazy annotation = methodMetadata.getIntrospectedMethod().getAnnotation(Lazy.class);
                if (null != annotation) {
                    return true;
                }
            } catch (NoSuchFieldException | IllegalAccessException | ClassCastException e) {
                logger.warn("Unexpected exception while processing bean definitions", e);
                return false;
            }
        }
        return false;
    }
}
