package ru.yandex.direct.common.tracing;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.AsyncEvent;
import javax.servlet.AsyncListener;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.CharMatcher;
import com.google.common.base.Strings;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.util.TraceCommentVars;
import ru.yandex.direct.tracing.util.TraceCommentVarsHolder;
import ru.yandex.direct.tracing.util.TraceUtil;

import static java.util.Collections.emptyMap;

@Component
public class TraceContextFilter extends OncePerRequestFilter {
    private TraceHelper traceHelper;
    private Map<String, String> traceTags;

    private static final CharMatcher DIGITS_MATCHER = CharMatcher.inRange('0', '9');
    // если урл начинается с таких префиксов - добавляем http method к концу
    private static final Set<String> HTTP_METHOD_AWARE_PREFIXES = Set.of("uac");
    // если урл начинается с таких префиксов - скрываем нечисловой ID в конце
    private static final Set<String> HIDE_HASH_PREFIXES = Set.of("uac.content.hash.");

    @Autowired
    public TraceContextFilter(TraceHelper traceHelper, DirectConfig tracing) {
        this.traceHelper = traceHelper;
        DirectConfig tracingConfig = tracing.getBranch("tracing");
        this.traceTags = tracingConfig.hasPath("tags") ? tracingConfig.getBranch("tags").asMap() : emptyMap();
    }

    @NotNull
    private String extractMethod(HttpServletRequest request) {
        // для основной части контроллеров, используем путь относительно servlet-path,
        // а если он пустой (как в случае /alive) - используем getServletPath
        String pathInfo = request.getPathInfo();
        if (Strings.isNullOrEmpty(pathInfo)) {
            pathInfo = request.getServletPath();
        }

        var parts = Arrays.stream(pathInfo.split("/"))
                .filter(s -> !s.isEmpty() && !s.equals("web-api"))
                .map(s -> DIGITS_MATCHER.matchesAllOf(s) ? "ID" : s)
                .collect(Collectors.toList());
        var method = StringUtils.join(parts, ".");
        for (String prefix : HIDE_HASH_PREFIXES) {
            if (method.startsWith(prefix)) {
                method = prefix + "ID";
            }
        }
        if (!parts.isEmpty() && HTTP_METHOD_AWARE_PREFIXES.contains(parts.get(0))) {
            return method + "." + request.getMethod();
        } else {
            return method;
        }
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        String header = request.getHeader(TraceUtil.X_YANDEX_TRACE);

        String method = extractMethod(request);

        try (TraceGuard guard = traceHelper.guardFromHeader(header, method)) {
            try {
                Map<String, String> traceCommentVars = TraceCommentVars.parseHeader(
                        request.getHeader(TraceCommentVars.X_YANDEX_TRACE_COMMENT_VARS));
                if (traceCommentVars.containsKey(TraceCommentVars.OPERATOR_FIELD)) {
                    TraceCommentVarsHolder.get().setOperator(traceCommentVars.get(TraceCommentVars.OPERATOR_FIELD));
                }
                TraceCommentVarsHolder.get().addTags(traceTags);
                Map<String, String> pageInfoVars = TraceCommentVars.parsePageInfoHeader(
                        request.getHeader(TraceCommentVars.X_PAGE_INFO));
                if (pageInfoVars.containsKey(TraceCommentVars.PAGE_REQID_FIELD)) {
                    TraceCommentVarsHolder.get().setPageReqid(pageInfoVars.get(TraceCommentVars.PAGE_REQID_FIELD));
                }
                if (pageInfoVars.containsKey(TraceCommentVars.PAGE_CMD_FIELD)) {
                    TraceCommentVarsHolder.get().setPageCmd(pageInfoVars.get(TraceCommentVars.PAGE_CMD_FIELD));
                }
                response.setHeader(AccelInfoHeader.HEADER_NAME, new AccelInfoHeader(emptyMap()).toString());
                filterChain.doFilter(request, response);
            } finally {
                if (request.isAsyncStarted()) {
                    // В случае асинхронной обработки откладываем трейс на более поздний срок
                    request.getAsyncContext().addListener(new AsyncListener() {
                        @Override
                        public void onComplete(AsyncEvent event) throws IOException {
                            guard.finish();
                            TraceCommentVarsHolder.get().removeOperator();
                            TraceCommentVarsHolder.get().removePageReqid();
                            TraceCommentVarsHolder.get().removePageCmd();
                        }

                        @Override
                        public void onTimeout(AsyncEvent event) throws IOException {
                            // нам не интересны таймауты
                        }

                        @Override
                        public void onError(AsyncEvent event) throws IOException {
                            // нам не интересны ошибки
                        }

                        @Override
                        public void onStartAsync(AsyncEvent event) throws IOException {
                            // Новый виток startAsync, переподписываемся на новый контекст
                            // Этот метод вызывается, если текущий запрос завершается через dispatch и другой
                            // обработчик снова вызывает startAsync. При этом старый контекст выкидывается и без
                            // переподписки мы не сможем определить окончание обработки запроса.
                            event.getAsyncContext().addListener(this);
                        }
                    });
                    guard.postpone();
                } else {
                    TraceCommentVarsHolder.get().removeOperator();
                    TraceCommentVarsHolder.get().removePageReqid();
                    TraceCommentVarsHolder.get().removePageCmd();
                }
            }
        }
    }
}
