package ru.yandex.direct.common.util;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.ThreadFactoryBuilder;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * ThreadPoolExecutor с фиксированным количеством тредов и размером очереди,
 * добавляющий статистику в {@code SOLOMON_REGISTRY},
 * а также оборачивающий запуск команд в текущий трейсинг.
 */
@ParametersAreNonnullByDefault
public class DirectThreadPoolExecutor extends ThreadPoolExecutor {
    private static final String METRIC_TOTAL_TASKS = "completedTaskCount";
    private static final String METRIC_ACTIVE_TASKS = "activeCount";
    private static final String METRIC_QUEUE_SIZE = "queueSize";

    public DirectThreadPoolExecutor(int poolSize, int queueSize, String threadPrefix) {
        super(poolSize, poolSize, 0L, TimeUnit.SECONDS, new ArrayBlockingQueue<>(queueSize),
                createFactory(threadPrefix));

        MetricRegistry registry = SOLOMON_REGISTRY.subRegistry("ThreadPoolExecutor", threadPrefix);
        registry.lazyCounter(METRIC_TOTAL_TASKS, this::getCompletedTaskCount);
        registry.lazyGaugeInt64(METRIC_ACTIVE_TASKS, this::getActiveCount);
        registry.lazyGaugeInt64(METRIC_QUEUE_SIZE, getQueue()::size);
    }

    private static ThreadFactory createFactory(String prefix) {
        return new ThreadFactoryBuilder().setNameFormat(prefix + "-%d").setDaemon(true).build();
    }

    @PreDestroy
    @Override
    public void shutdown() {
        super.shutdown();
    }

    @Override
    public void execute(Runnable command) {
        super.execute(Trace.current().wrap(command));
    }
}
