package ru.yandex.direct.common.util;

import java.util.EnumMap;
import java.util.Map;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collector;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

/**
 * Вспомогательные collect-ры для работы с guava-ими коллекциями
 */
public class GuavaCollectors {
    private GuavaCollectors() {
    }

    /**
     * Преобразовать stream в SetMultimap
     */
    @SuppressWarnings("CheckReturnValue")
    public static <T, K, U> Collector<T, ?, Multimap<K, U>> toMultimap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
        return Collector.of(
                HashMultimap::<K, U>create,
                (acc, t) -> acc.put(keyMapper.apply(t), valueMapper.apply(t)),
                (map1, map2) -> {
                    map1.putAll(map2);
                    return map1;
                },
                Collector.Characteristics.UNORDERED,
                Collector.Characteristics.IDENTITY_FINISH);
    }

    /**
     * Преобразовать stream Entry в ListMultimap
     */
    @SuppressWarnings("CheckReturnValue")
    public static <K, V> Collector<Map.Entry<K, ? extends Iterable<V>>, ?, ListMultimap<K, V>> toListMultimap() {
        return Collector.of(
                ArrayListMultimap::create,
                (acc, t) -> acc.putAll(t.getKey(), t.getValue()),
                (map1, map2) -> {
                    map1.putAll(map2);
                    return map1;
                });
    }

    /**
     * Преобразовать stream в ListMultimap
     */
    @SuppressWarnings("CheckReturnValue")
    public static <T, K, U> Collector<T, ?, ListMultimap<K, U>> toListMultimap(
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
        return Collector.of(
                ArrayListMultimap::<K, U>create,
                (acc, t) -> acc.put(keyMapper.apply(t), valueMapper.apply(t)),
                (map1, map2) -> {
                    map1.putAll(map2);
                    return map1;
                },
                Collector.Characteristics.IDENTITY_FINISH);
    }

    /**
     * Преобразовать stream в ImmutableEnumMap
     */
    public static <T, K extends Enum<K>, U> Collector<T, ?, EnumMap<K, U>> toImmutableEnumMap(
            Class<K> keyClazz,
            Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
        BinaryOperator<U> throwingMerger = MoreBinaryOperators.throwingMerger();
        return Collector.of(
                () -> Maps.newEnumMap(keyClazz),
                (map, t) -> map.merge(keyMapper.apply(t), valueMapper.apply(t), throwingMerger),
                MoreBinaryOperators.mapMerger(throwingMerger),
                Collector.Characteristics.IDENTITY_FINISH);
    }
}
