package ru.yandex.direct.common.util;

import java.util.Optional;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import com.google.common.net.InternetDomainName;

import ru.yandex.direct.common.enums.YandexDomain;

import static ru.yandex.direct.common.util.HttpUtil.getRequest;

public class HostUtils {
    private HostUtils() {
    }

    /**
     * Возвращает хост клиента с которого был сделан запрос
     * Если хост получить не удалось, возвращает Optional.empty()
     *
     * @return хост клиента без порта
     */
    public static Optional<String> getRemoteHost() {
        Optional<String> result;
        try {
            result = Optional.ofNullable(getRequest()).map(HttpServletRequest::getServerName);
        } catch (IllegalStateException e) {
            result = Optional.empty();
        }

        return result;
    }

    /**
     * Возвращает яндекс домен
     *
     * @see #getYandexDomain(String)
     */
    public static Optional<YandexDomain> getYandexDomain() {
        return getRemoteHost().flatMap(HostUtils::getYandexDomain);
    }

    /**
     * Возвращает яндекс домен
     * Например для хоста 13945.beta6.direct.yandex.ru вернет {@link YandexDomain#RU}
     *
     * @return яндекс домен
     */
    public static Optional<YandexDomain> getYandexDomain(@Nullable String host) {
        if (host == null) {
            return Optional.empty();
        }
        InternetDomainName domain = InternetDomainName.from(host);
        try {
            InternetDomainName internetDomainName = domain.topPrivateDomain();
            YandexDomain yandexDomain = YandexDomain.fromValue(internetDomainName.toString());
            return Optional.ofNullable(yandexDomain);
        } catch (IllegalStateException e) {
            return Optional.empty();
        }
    }

}
