package ru.yandex.direct.common.util;

import java.net.InetAddress;
import java.net.URL;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.Strings;
import com.google.common.net.HttpHeaders;
import one.util.streamex.StreamEx;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import ru.yandex.direct.common.net.IpUtils;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.net.LocalhostUtils;

import static org.springframework.util.MimeTypeUtils.APPLICATION_JSON_VALUE;
import static ru.yandex.direct.env.EnvironmentType.SANDBOX_TESTING;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;

public class HttpUtil {
    public static final String HEADER_AUTHORIZATION = "Authorization";
    public static final String HEADER_X_REAL_IP = "X-Real-IP";
    public static final String HEADER_X_PROXY_REAL_IP = "X-Proxy-Real-IP";
    private static final String HEADER_X_REAL_HOST = "X-Real-Host";
    private static final String INTERNAL_LOCATION_S3_FILE = "/serve_s3_file/";
    private static final String INTERNAL_LOCATION_DIRECT_FILES = "/serve_mds_file/";
    private static final Pattern PATTERN_MDS_URL = Pattern.compile("^http://(.*?)/get-direct-files/(.*?)$");

    public static final String BEARER_TOKEN_TYPE = "bearer";
    public static final String ACCEPT_HEADER = "Accept";
    public static final String XML_HTTP_REQUEST = "XMLHttpRequest";
    public static final String X_REQUESTED_WITH_HEADER = "X-Requested-With";
    public static final String YANDEX_GID = "yandex_gid";
    public static final String GDPR = "gdpr";
    public static final String IS_GDPR = "is_gdpr";
    public static final String YANDEXUID = "yandexuid";
    public static final String YP = "yp";
    public static final String X_ACCEL_REDIRECT = "X-Accel-Redirect";

    // Headers
    public static final String USER_AGENT_HEADER_NAME = "User-Agent";
    public static final String DETECTED_LOCALE_HEADER_NAME = "X-Detected-Locale";

    // временный заголовок для извлечения ip источника запроса,
    // который не перезатирается балансером Директа
    private static final String HEADER_SECRET_REAL_IP = "X-laerYllaer-IP";

    private HttpUtil() {
    }

    public static Optional<String> extractBearerToken(HttpServletRequest request, String bearerToken) {
        Enumeration<String> authHeaders = request.getHeaders(HEADER_AUTHORIZATION);
        while (authHeaders != null && authHeaders.hasMoreElements()) {
            String value = authHeaders.nextElement().trim();
            if (value.toLowerCase().startsWith(bearerToken.toLowerCase())) {
                value = value.substring(bearerToken.length()).trim();
                int commaIndex = value.indexOf(',');
                if (commaIndex > 0) {
                    value = value.substring(0, commaIndex).trim();
                }
                return Optional.ofNullable(Strings.emptyToNull(value));
            }
        }
        return Optional.empty();
    }

    public static Optional<String> extractBearerToken(HttpServletRequest request) {
        return extractBearerToken(request, BEARER_TOKEN_TYPE);
    }

    public static boolean acceptXProxyRealApi(@Nullable TvmService tvmService, EnvironmentType environmentType) {
        if (environmentType.isProductionOrPrestable()) {
            return tvmService == DIRECT_API_PROD;
        } else if (environmentType == SANDBOX_TESTING) {
            return tvmService == DIRECT_API_SANDBOX_TEST;
        } else if (environmentType.isSandbox()) {
            return tvmService == DIRECT_API_SANDBOX;
        } else if (environmentType.isTesting()) {
            return tvmService == DIRECT_API_TEST;
        } else if (environmentType.isDevelopment() // для юнит тестов
                || environmentType.isBeta()) {    // для того чтобы на бетах работало как на ТС и можно было дебажить
            return tvmService == DIRECT_API_TEST;
        } else {
            return false;
        }
    }

    /**
     * Возвращает адрес клиента либо последней машины, с которой был сделан запрос.
     * Адрес равен:
     * 1) http-заголовку 'X-Real-IP', если есть (не всегда является адресом клиента, может перезатираться балансером)
     * 2) httpServletRequest.getRemoteAddr() если заголовка нет
     *
     * @param httpServletRequest http-запрос клиента
     * @return адрес клиента
     */
    public static InetAddress getRemoteAddress(HttpServletRequest httpServletRequest) {
        String ip = getRemoteIp(httpServletRequest);
        return ip == null ? null : IpUtils.ipFromStringWithOptionalBrackets(ip);
    }

    public static String getRemoteIp(HttpServletRequest httpServletRequest) {
        return getHeaderValue(httpServletRequest, HEADER_X_REAL_IP).orElseGet(httpServletRequest::getRemoteAddr);
    }

    /**
     * Возвращает адрес клиента либо последней машины, с которой был сделан запрос.
     * Если адрес получить не удалось, возвращает Optional.empty()
     *
     * @return адрес клиента
     */
    public static Optional<InetAddress> getRemoteAddress() {
        Optional<InetAddress> result;
        try {
            result = Optional.of(getRequest()).map(HttpUtil::getRemoteAddress);
        } catch (IllegalStateException e) {
            result = Optional.empty();
        }

        return result;
    }

    public static InetAddress getSecretRemoteAddress(HttpServletRequest httpServletRequest) {
        String ip = getSecretRemoteIp(httpServletRequest);
        return ip == null ? null : IpUtils.ipFromStringWithOptionalBrackets(ip);
    }

    public static String getSecretRemoteIp() {
        return getSecretRemoteIp(getRequest());
    }

    public static String getSecretRemoteIp(HttpServletRequest request) {
        return getHeaderValue(request, HEADER_SECRET_REAL_IP).orElse(getRemoteIp(request));
    }

    /**
     * Получить Host заголовок запроса
     */
    @Nullable
    public static String getExactHostHeader() {
        return getHeaderValue(getRequest(), HttpHeaders.HOST).orElse(null);
    }


    /**
     * Получить Host заголовок от прокси или родной
     */
    public static Optional<String> getHostHeader(HttpServletRequest request) {
        return getHeaderValue(request, HEADER_X_REAL_HOST)
                .or(() -> getHeaderValue(request, HttpHeaders.HOST));
    }

    /**
     * Возвращает адрес клиента, или LocalhostUtils.localAddress()
     */
    public static IpAddress getRemoteAddressForBlackbox() {
        return getRemoteAddress().map(IpAddress::valueOf).orElse(LocalhostUtils.localAddress());
    }

    public static Optional<String> getHeaderValue(HttpServletRequest request, String headerName) {
        String value = request.getHeader(headerName);
        if (value == null) {
            return Optional.empty();
        } else {
            return Optional.ofNullable(Strings.emptyToNull(value.trim()));
        }
    }

    public static boolean isHeaderEqualsToValue(HttpServletRequest request, String headerName, String value) {
        String realValue = request.getHeader(headerName);
        return realValue != null && realValue.trim().equalsIgnoreCase(value.trim());
    }

    public static String getRequestUrl(HttpServletRequest request) {
        String requestUrl = request.getRequestURL().toString();
        if (request.getQueryString() != null) {
            requestUrl += "?" + request.getQueryString();
        }
        return requestUrl;
    }

    /**
     * @return true если в запросе есть Http-заголовок {@link #X_REQUESTED_WITH_HEADER} со значением
     * {@link #XML_HTTP_REQUEST} или {@link #ACCEPT_HEADER}
     * со значением содержащим {@link org.springframework.util.MimeTypeUtils#APPLICATION_JSON_VALUE}, иначе false
     */
    public static boolean isAjax(HttpServletRequest request) {
        boolean hasApplicationJsonTypeInAcceptHeader = false;
        if (request.getHeader(ACCEPT_HEADER) != null) {
            hasApplicationJsonTypeInAcceptHeader = request.getHeader(ACCEPT_HEADER).contains(APPLICATION_JSON_VALUE);
        }
        return XML_HTTP_REQUEST.equals(request.getHeader(X_REQUESTED_WITH_HEADER))
                || hasApplicationJsonTypeInAcceptHeader;
    }

    /**
     * Wrapper для {@link RequestContextHolder#currentRequestAttributes} + {@link ServletRequestAttributes#getRequest()}
     * Требует наличия {@link org.springframework.web.servlet.DispatcherServlet},
     * либо {@link  org.springframework.web.filter.RequestContextFilter}.
     * Последнее требуется, к примеру, для фильтров аутентификации, т.к. они работают
     * до {@link org.springframework.web.servlet.DispatcherServlet}.
     *
     * @see RequestContextHolder
     */
    public static HttpServletRequest getRequest() {
        return ((ServletRequestAttributes) RequestContextHolder.currentRequestAttributes()).getRequest();
    }

    /**
     * Проверяет, привязан ли текущий поток к http-запросу
     */
    public static boolean isRequestExists() {
        return RequestContextHolder.getRequestAttributes() != null;
    }

    /**
     * Возвращает первую cookie с именем name, если она есть
     */
    public static Optional<String> getCookieValue(String name, HttpServletRequest httpRequest) {
        Cookie[] cookies = httpRequest.getCookies();
        if (cookies == null) {
            return Optional.empty();
        }
        return StreamEx.of(cookies)
                .findFirst(cookie -> cookie.getName().equals(name)).map(Cookie::getValue);
    }

    /**
     * Возвращает geoRegionId текущего запроса {@link #getRequest()}
     *
     * @see #getGeoRegionId(HttpServletRequest)
     */
    public static Optional<Long> getCurrentGeoRegionId() {
        var request = getRequest();
        return getCurrentGeoRegionId(request);
    }

    public static Optional<Long> getCurrentGeoRegionId(HttpServletRequest request) {
        Optional<Long> geoRegionId;
        try {
            geoRegionId = getGeoRegionId(request);
        } catch (NumberFormatException e) {
            geoRegionId = Optional.empty();
        }

        return geoRegionId;
    }

    /**
     * Возвращает значение куки {@link #YANDEX_GID}
     * TODO DIRECT-76463: Добавить получение regionId по ip - перловый метод HttpTools::http_geo_exact_region
     * <p>
     * См. perl HttpTools::http_geo_exact_region
     */
    private static Optional<Long> getGeoRegionId(HttpServletRequest request) throws NumberFormatException {
        return getCookieValue(YANDEX_GID, request).map(Long::valueOf);
    }

    public static Optional<Long> getGdprCookie(HttpServletRequest request) {
        return getCookieValue(GDPR, request).map(Long::valueOf);
    }

    public static Optional<Long> getIsGdprCookie(HttpServletRequest request) {
        return getCookieValue(IS_GDPR, request).map(Long::valueOf);
    }

    public static Optional<String> getYandexUid(HttpServletRequest request) {
        return getCookieValue(YANDEXUID, request);
    }

    public static Optional<String> getYpCookie(HttpServletRequest request) {
        return getCookieValue(YP, request);
    }

    /**
     * Возвращает локаль текущего запроса {@link #getRequest()}
     *
     * @see #getLocale(HttpServletRequest)
     */
    public static Optional<Locale> getCurrentLocale() {
        Optional<Locale> locale;
        try {
            locale = getLocale(getRequest());
        } catch (RuntimeException e) {
            locale = Optional.empty();
        }

        return locale;
    }

    /**
     * Возвращает локаль запроса, который ожидаем найти в атрибутах запроса {@link #DETECTED_LOCALE_HEADER_NAME}
     */
    private static Optional<Locale> getLocale(HttpServletRequest request) {
        return Optional.ofNullable(request.getAttribute(DETECTED_LOCALE_HEADER_NAME))
                .map(Locale.class::cast);
    }

    /**
     * Выставить заголовок для внутреннего (nginx) редиректа на обработчик файлов MDS-S3
     *
     * @param response ответ сервера
     * @param url      url на который треубется выставить внутренний редирект
     */
    public static void setInternalRedirectToS3(HttpServletResponse response, URL url) {
        response.setHeader(HttpUtil.X_ACCEL_REDIRECT, INTERNAL_LOCATION_S3_FILE + url.getHost() + url.getPath());
    }

    /**
     * Выставить заголовок для внутреннего (nginx) редиректа на обработчик файлов MDS direct-files
     *
     * @param response    ответ сервера
     * @param mdsHost     хост MDS
     * @param mdsFilePath путь к файлу в mds
     * @param customName  имя файла
     */
    public static void setInternalRedirectToMdsDirectFiles(
            HttpServletResponse response, String mdsHost, String mdsFilePath, @Nullable String customName) {
        String value = INTERNAL_LOCATION_DIRECT_FILES + mdsHost + "/" + mdsFilePath
                + "?content_type=application/octet-stream";
        response.setHeader(HttpUtil.X_ACCEL_REDIRECT, value);
        var fileName = Optional.ofNullable(customName).orElse(mdsFilePath);
        response.setHeader(HttpHeaders.CONTENT_DISPOSITION, String.format("attachment; filename=\"%s\"", fileName));
    }

    /**
     * Выставить заголовок для внутреннего (nginx) редиректа на обработчик файлов MDS direct-files
     *
     * @param response   ответ сервера
     * @param mdsUrl     url в mds
     * @param customName имя файла
     */
    public static void setInternalRedirectToMdsDirectFiles(
            HttpServletResponse response, String mdsUrl, @Nullable String customName) {
        Matcher matcher = PATTERN_MDS_URL.matcher(mdsUrl);
        if (matcher.find()) {
            setInternalRedirectToMdsDirectFiles(response, matcher.group(1), matcher.group(2), customName);
        } else {
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
    }
}
