package ru.yandex.direct.common.util;

import java.util.Locale;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.context.i18n.LocaleContextHolder;

import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Language;

/**
 * Try-with-resources класс для восстановления исходной локали в {@link LocaleContextHolder}.
 * <p>
 * Новое и предыдущее значения локали устанавливаются с флагом "наследовать" (дочерними тредами)
 */
@ParametersAreNonnullByDefault
public class LocaleGuard implements AutoCloseable {

    private final Locale previousLocale;

    /**
     * Запоминает текущую локаль и установливает новое значение.
     *
     * @param newLocale новая локаль
     */
    public LocaleGuard(Locale newLocale) {
        previousLocale = LocaleContextHolder.getLocale();
        LocaleContextHolder.setLocale(newLocale, true);
    }

    /**
     * Задать локаль в контексте по языку.
     * Не учитывает региональных настроек (регион, таймзона), но хорошо стыкуется с нашим механизмом переводов
     *
     * @param language язык
     * @return созданный объект {@link LocaleGuard}
     * @throws IllegalStateException если для языка нет предопределенной локали
     */
    public static LocaleGuard fromLanguage(Language language) {
        switch (language) {
            case EN:
                return new LocaleGuard(I18NBundle.EN);
            case TR:
                return new LocaleGuard(I18NBundle.TR);
            case RU:
                return new LocaleGuard(I18NBundle.RU);
            case UK:
                return new LocaleGuard(I18NBundle.UA);
            default:
                throw new IllegalStateException("No predefined Locale for Language: " + language.getLangString());
        }
    }

    @Override
    public void close() {
        LocaleContextHolder.setLocale(previousLocale, true);
    }
}
