package ru.yandex.direct.common.util;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import static org.apache.commons.beanutils.BeanUtils.copyProperty;

@Component
public class PropertyFilter {

    public static class PropertiesDescritor {
        public Set<String> describe(Object bean) {
            try {
                Map<String, String> tmp = BeanUtils.describe(bean);
                return tmp.keySet();
            } catch (Exception e) {
                throw new IllegalArgumentException("can not describe bean properties", e);
            }
        }
    }

    private ConcurrentHashMap<Class<?>, Set<String>> propertiesCache = new ConcurrentHashMap<>();
    private PropertiesDescritor propertiesDescritor = new PropertiesDescritor();

    public void filterProperties(Object bean, List<String> propsToLeave) {
        Assert.notNull(bean, "bean for filtering is required");
        Assert.notNull(propsToLeave, "properties to leave is required");
        try {
            Set<String> propsToDelete = getBeanProperties(bean);
            propsToDelete.remove("class");
            propsToDelete.removeAll(propsToLeave);
            for (String property : propsToDelete) {
                copyProperty(bean, property, null);
            }
        } catch (Exception e) {
            throw new IllegalArgumentException("can not clear properties", e);
        }
    }

    private Set<String> getBeanProperties(Object bean) {
        Set<String> cachedProperties = propertiesCache.get(bean.getClass());
        if (cachedProperties == null) {
            cachedProperties = cacheBeanProperties(bean);
        }
        // copying is required because cached bean properties set must be immutable
        return new HashSet<>(cachedProperties);
    }

    private Set<String> cacheBeanProperties(Object bean) {
        Function<Class<?>, Set<String>> beanPropertiesProvider =
                cls -> ImmutableSet.copyOf(propertiesDescritor.describe(bean));
        return propertiesCache.computeIfAbsent(bean.getClass(), beanPropertiesProvider);
    }

    void setPropertiesDescritor(PropertiesDescritor propertiesDescritor) {
        this.propertiesDescritor = propertiesDescritor;
    }
}
