package ru.yandex.direct.common.util;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;

/**
 * Помощник для засыпания на время, кратное времени выполнения метода
 * <pre> {@code
 * RelaxedWorker worker = new RelaxedWorker(2.0);
 * worker.runAndRelax(() -> { ... do something ... });
 * int result = worker.callAndRelax(() -> { ... do something ...; return xxx; })
 * }</pre>
 */
public class RelaxedWorker {
    private static final Logger logger = LoggerFactory.getLogger(RelaxedWorker.class);

    @SuppressWarnings("WeakerAccess")
    protected static final double DEFAULT_SLEEP_COEFFICIENT = 1.0;

    private final double sleepCoefficient;

    /**
     * Конструктор с возможностью указать коэффициент сна
     *
     * @param sleepCoefficient во сколько раз больше спать по сравнению с временем выполнения метода
     * @throws IllegalArgumentException при отрицательном значении {@code sleepCoefficient}
     */
    public RelaxedWorker(double sleepCoefficient) {
        if (sleepCoefficient < 0) {
            throw new IllegalArgumentException("sleepCoefficient cannot be less than 0");
        }
        this.sleepCoefficient = sleepCoefficient;
    }

    /**
     * Конструктор с умолчательным коэффициентом сна, равным {@link #DEFAULT_SLEEP_COEFFICIENT}
     */
    public RelaxedWorker() {
        this(DEFAULT_SLEEP_COEFFICIENT);
    }

    private void sleep(Stopwatch stopwatch) {
        long durationMillis = (long) (stopwatch.elapsed(TimeUnit.MILLISECONDS) * sleepCoefficient);

        if (durationMillis > 0) {
            logger.debug("sleeping for {} ms", durationMillis);
            try {
                Trace.sleep(durationMillis);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(e);
            }
        }
    }

    /**
     * Выполняет {@code method} и усыпляет текущий тред на время в {@link #sleepCoefficient} раз превышающее
     * время выполнения метода
     *
     * @param <E>    тип исключения, которое бросает {@code method}
     * @param method UnsafeRunnable функция для выполнения
     * @throws E исключение, брошенное {@code method}
     */
    public <E extends Exception> void runAndRelax(@Nonnull UnsafeRunnable<E> method) throws E {
        Stopwatch stopwatch = Stopwatch.createStarted();
        method.run();
        sleep(stopwatch.stop());
    }

    /**
     * Выполняет {@code method} и усыпляет текущий тред на время в {@link #sleepCoefficient} раз превышающее
     * время выполнения метода
     *
     * @param <V>    Тип возвращаемого значения, возвращаемого {@code method}
     * @param <E>    тип исключения, которое бросает {@code method}
     * @param method UnsafeCallable функция для выполнения
     * @throws E исключение, брошенное {@code method}
     */
    public <V, E extends Exception> V callAndRelax(@Nonnull UnsafeCallable<V, E> method) throws E {
        Stopwatch stopwatch = Stopwatch.createStarted();
        V result = method.call();
        sleep(stopwatch.stop());

        return result;
    }

    /**
     * Задача, которая не возвращает результата и может бросать исключение.
     * Требует определениея единственного метода без аргументов, называющегося
     * {@code run}.
     *
     * @param <E> Тип исключения, которое может бросать {@code run}
     * @see java.lang.Runnable
     */
    @FunctionalInterface
    public interface UnsafeRunnable<E extends Exception> {
        void run() throws E;
    }

    /**
     * Задача, которая возвращает результат и может бросать исключение.
     * Требует определениея единственного метода без аргументов, называющегося
     * {@code call}.
     *
     * @param <V> Тип возвращаемого значения
     * @param <E> Тип исключения, которое может бросать {@code call}
     * @see java.util.concurrent.Callable
     */
    @FunctionalInterface
    public interface UnsafeCallable<V, E extends Exception> {
        V call() throws E;
    }
}
