package ru.yandex.direct.common.util;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;

import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.read.ReaderFunction1;
import ru.yandex.direct.jooqmapper.write.WriterFunction1;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.mysqlcompression.MysqlCompression;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbutil.SqlUtils.mysqlZeroLocalDate;
import static ru.yandex.direct.dbutil.SqlUtils.mysqlZeroLocalDateTime;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RepositoryUtils {
    public static final Long TRUE = 1L;
    public static final Long FALSE = 0L;
    public static final String TRUE_ENUM = "Yes";
    public static final String FALSE_ENUM = "No";
    //Значение, которое можно использовать как плейсхолдер для проставления текущего времени при записи в базу
    public static final LocalDateTime NOW_PLACEHOLDER = LocalDateTime.MIN;

    private static final String MYSQL_SET_DELIMITER = ",";
    private static final String MYSQL_SET_DELIMITER_REGEX = "\\s*,\\s*";
    private static final String EMPTY_STRING = "";
    private static final int MYSQL_MEDIUMBLOB_LENGTH = (1 << 24) - 1;

    private RepositoryUtils() {
    }

    public static <T extends Enum<T>> T booleanToYesNo(Boolean bool, Class<T> cls) {
        return booleanToYesNoWithDefault(bool, cls, null);
    }

    public static <T extends Enum<T>> T booleanToYesNoWithDefault(Boolean bool, Class<T> cls, T defaultValue) {
        return bool != null ? (bool ? Enum.valueOf(cls, TRUE_ENUM) : Enum.valueOf(cls, FALSE_ENUM)) : defaultValue;
    }

    public static <T extends Enum<T>> Boolean booleanFromYesNo(T t) {
        return t != null ? t.name().equals(TRUE_ENUM) : null;
    }

    public static Long booleanToLong(Boolean bool) {
        return bool != null ? (bool ? TRUE : FALSE) : null;
    }

    public static Long nullSafeBooleanToLong(Boolean bool) {
        return nullSafeBooleanWithDefault(bool, FALSE);
    }

    public static Long nullSafeBooleanToLongDefaultTrue(Boolean bool) {
        return nullSafeBooleanWithDefault(bool, TRUE);
    }

    private static Long nullSafeBooleanWithDefault(Boolean bool, Long defaultValue) {
        return bool != null ? (bool ? TRUE : FALSE) : defaultValue;
    }

    public static Boolean booleanFromLong(Long l) {
        return l != null ? (l == 1L) : null;
    }

    public static Long intToLong(Integer integer) {
        return integer != null ? Long.valueOf(integer) : null;
    }

    public static Integer intFromLong(Long l) {
        return l != null ? l.intValue() : null;
    }

    public static Long shortToLong(Short sh) {
        return sh != null ? Long.valueOf(sh) : null;
    }

    public static Short shortFromLong(Long l) {
        return l != null ? l.shortValue() : null;
    }

    public static ULong bigIntegerToULong(BigInteger bigInteger) {
        return bigInteger != null ? ULong.valueOf(bigInteger) : null;
    }

    public static BigInteger bigIntegerFromULong(ULong unLong) {
        return unLong != null ? unLong.toBigInteger() : null;
    }

    public static BigDecimal percentToBigInteger(Percent percent) {
        return percent != null ? percent.asPercent() : null;
    }

    public static Percent percentFromBigInteger(BigDecimal bigDecimal) {
        return bigDecimal != null ? Percent.fromPercent(bigDecimal) : null;
    }

    public static Long currencyToDb(CurrencyCode currencyCode) {
        return currencyCode != null ?
                currencyCode.getCurrency().getIsoNumCode().longValue() : null;
    }

    public static CurrencyCode currencyFromDb(Long currencyIsoCode) {
        return currencyIsoCode != null ?
                Currencies.getCurrency(currencyIsoCode.intValue()).getCode() : null;
    }

    /**
     * {@code null-safe} обёртка для функции {@code func}, обеспечивающая контракт
     * <pre> if arg == null then null else func(arg) </pre>
     */
    public static <T, V> Function<T, V> nullSafeWrapper(Function<T, V> func) {
        return t -> t != null ? func.apply(t) : null;
    }

    /**
     * {@code null-safe} обёртка для читающей функции {@code func}, обеспечивающая контракт
     * <pre> if arg == null then null else func(arg) </pre>
     */
    public static <T, V> ReaderFunction1<T, V> nullSafeReader(ReaderFunction1<T, V> func) {
        return t -> t != null ? func.read(t) : null;
    }

    /**
     * {@code null-safe} обёртка для пишущей функции {@code func}, обеспечивающая контракт
     * <pre> if arg == null then null else func(arg) </pre>
     */
    public static <T, V> WriterFunction1<T, V> nullSafeWriter(WriterFunction1<T, V> func) {
        return t -> t != null ? func.write(t) : null;
    }

    /**
     * Сконвертировать сет из значений в его представление в базе данных MySQL (список строк через запятую)
     *
     * @param set           сет в представлении модели
     * @param elementMapper как записывать элемент множества в строку
     * @param <T>           тип обрабатываемого значения
     */
    public static <T> String setToDb(Set<T> set, Function<T, String> elementMapper) {
        return set != null ? String.join(MYSQL_SET_DELIMITER, mapList(set, elementMapper)) : null;
    }

    /**
     * Сконвертировать строку с представлением сета в базе данных MySQL (список строк через запятую) в сет значений
     *
     * @param str           сет в представлении базы данных
     * @param elementMapper как превратить строку из базы в enum
     * @param <T>           тип обрабатываемого значения
     */
    public static <T> Set<T> setFromDb(String str, Function<String, T> elementMapper) {
        return str != null && !EMPTY_STRING.equals(str.trim()) ?
                StreamEx.split(str, MYSQL_SET_DELIMITER_REGEX)
                        .map(elementMapper)
                        .collect(Collectors.toSet())
                : Collections.emptySet();
    }

    public static Set<String> setFromDb(String str) {
        return setFromDb(str, Function.identity());
    }

    /**
     * Получить reader фунцию для чтения из mysql-Set поля в EnumSet.
     * {@code null} значения превращает в пустое множество
     *
     * @param enumCls    - класс enum
     * @param enumMapper маппер из строк в значения enum
     * @param <T>        тип enum
     */
    public static <T extends Enum<T>> ReaderFunction1<String, Set<T>> enumSetReader(Class<T> enumCls,
                                                                                    Function<String, T> enumMapper) {
        return setString -> {
            EnumSet<T> result = EnumSet.noneOf(enumCls);

            if (setString == null) {
                return result;
            }

            StreamEx.split(setString, MYSQL_SET_DELIMITER)
                    .map(String::trim)
                    .filter(StringUtils::isNotEmpty)
                    .map(enumMapper)
                    .forEach(result::add);

            return result;
        };
    }

    /**
     * Сериализовать объект в JSON-строку и сжать совместимыми с функцией MySQL {@code uncompress()}.
     *
     * @param object    объект для сериализации
     * @param maxLength длина поля в базе данных для проверки размера сжатых данных
     * @param <T>       тип обрабатываемого объекта
     * @return массив байт или {@code null}, если объект был равен {@code null}
     * @throws IllegalArgumentException если сжатые данные превышают {@code maxLength}
     */
    public static <T> byte[] toCompressedJsonDb(T object, int maxLength) {
        if (object == null) {
            return null;
        }

        byte[] result = MysqlCompression.compress(JsonUtils.toJsonBytes(object));
        checkArgument(result.length <= maxLength, "packed value too long: %s > %s bytes", result.length, maxLength);
        return result;
    }

    /**
     * Сериализовать объект в JSON-строку и сжать совместимыми с функцией MySQL {@code uncompress()}.
     *
     * @param object объект для сериализации
     * @param <T>    тип обрабатываемого объекта
     * @return массив байт или {@code null}, если объект был равен {@code null}
     * @throws IllegalArgumentException если сжатые данные превышают размер колонки типа MEDIUMBLOB
     */
    public static <T> byte[] toCompressedJsonMediumblobDb(T object) {
        return toCompressedJsonDb(object, MYSQL_MEDIUMBLOB_LENGTH);
    }

    /**
     * Десериализовать сжатые функцией MySQL {@code compress()} json-данные к объект
     *
     * @param bytes сжатые данные
     * @param cls   класс объекта
     * @param <T>   тип обрабатываемого объекта
     * @return десериализованыеей объект или {@code null}, если сжатые данные были {@code null}
     * @throws ru.yandex.direct.mysqlcompression.MysqlCompressionException, если формат сжатых данных неправильный
     */
    public static <T> T objectFromCompressedJsonDb(byte[] bytes, Class<T> cls) {
        if (bytes == null) {
            return null;
        }
        return JsonUtils.fromJson(MysqlCompression.uncompress(bytes), cls);
    }

    /**
     * Конвертирует nullable дату в поле для записи в MySQL по правилу:
     * <pre> if arg == null then '0000-00-00' else arg </pre>
     *
     * @param date дата для конвертации
     * @return поле для записи в MySQL
     */
    public static Field<LocalDate> zeroableDateToDb(@Nullable LocalDate date) {
        if (date == null) {
            return mysqlZeroLocalDate();
        } else {
            return DSL.value(date).cast(LocalDate.class);
        }
    }

    /**
     * Конвертирует nullable дату в поле для записи в MySQL по правилу:
     * <pre> if arg == null then '0000-00-00 00:00:00' else arg </pre>
     *
     * @param dateTime дата для конвертации
     * @return поле для записи в MySQL
     */
    public static Field<LocalDateTime> zeroableDateTimeToDb(@Nullable LocalDateTime dateTime) {
        if (dateTime == null) {
            return mysqlZeroLocalDateTime();
        } else {
            return DSL.value(dateTime).cast(LocalDateTime.class);
        }
    }

    /**
     * Конвертирует nullable строку в значение для not-null строковой записи вMySQL по правилу:
     * <pre> if arg == null then '' else arg </pre>
     *
     * @param string строка для конвертации
     * @return поле для записи в MySQL
     */
    public static String nullableStringToNotNullDbField(@Nullable String string) {
        return Objects.requireNonNullElse(string, "");
    }

    /**
     * Конвертирует значение not-null строковой записи в MySQL в nullable-строку по правилу:
     * <pre> if arg == '' then null else arg </pre>
     *
     * @param string строка для конвертации
     * @return nullable значение записи
     */
    @Nullable
    public static String nullableStringFromNotNullDbField(String string) {
        if (string.equals("")) {
            return null;
        } else {
            return string;
        }
    }

    /**
     * Проставляет текущее время, если в качестве значения передан особый плейсхолдер
     * Иначе оставляет переданное значение
     *
     * @param dateTime — дата для проверки
     * @return nullable значение записи
     */
    @Nullable
    public static Field<LocalDateTime> setCurrentLocalDateTimeIfShould(@Nullable LocalDateTime dateTime) {
        if (NOW_PLACEHOLDER.equals(dateTime)) {
            return DSL.currentLocalDateTime();
        } else {
            return DSL.value(dateTime).cast(LocalDateTime.class);
        }
    }

    /**
     * Добавляет или обновляет строку в таблице, но только если appliedChanges затрагивают её поля.
     * Используется для моделей распределённых по нескольким таблицам, у которых записи в разных таблицах заводятся по
     * мере надобности.
     * Пример: модель User и её вспомогательные таблицы users_options и internal_users.
     *
     * @param context        контекст выполнения запроса.
     * @param appliedChanges Изменения, которые надо применить к соотвествующим записям в таблице.
     *                       Если в таблице ещё нет записей, то они будут созданы по модели полученной из
     *                       appliedChanges.
     *                       ЭТО ВАЖНО!!! Ожидается, что в appliedChanges консистентная модель.
     * @param mapper         Описание конвертации полей из модели ядра в БД.
     * @param idField        Поле с внешним ключом в таблице, в которой нужно изменить или вставить записи.
     * @param <R>            Тип модели записей базы.
     * @param <M>            Тип модели ядра.
     */
    public static <R extends Record, M extends ModelWithId> void updateOrInsert(
            DSLContext context,
            Collection<AppliedChanges<M>> appliedChanges,
            JooqMapperWithSupplier<M> mapper,
            TableField<R, Long> idField) {
        Table<R> table = idField.getTable();
        List<M> modelsInTable = new ArrayList<>();
        List<AppliedChanges<M>> changesInTable = new ArrayList<>();
        for (var ac : appliedChanges) {
            M model = ac.getModel();
            Set<ModelProperty<? super M, ?>> propertiesForUpdate = ac.getPropertiesForUpdate();
            Map<TableField<R, ?>, ?> dbFieldValues = mapper.getDbFieldValues(model, table, propertiesForUpdate);
            if (dbFieldValues.isEmpty()) {
                continue;
            }
            modelsInTable.add(model);
            changesInTable.add(ac);
        }
        List<Long> insertedIds = new InsertHelper<>(context, table)
                .addAll(mapper, modelsInTable)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAddedAndReturn(idField);
        Set<Long> fastSearchingIds = new HashSet<>(insertedIds);
        List<AppliedChanges<M>> restChanges = changesInTable.stream()
                .filter(ac -> !fastSearchingIds.contains(ac.getModel().getId()))
                .collect(toList());
        new UpdateHelper<>(context, idField)
                .processUpdateAll(mapper, restChanges)
                .execute();
    }

    public static Long nullToZero(@Nullable Long val) {
        return nvl(val, 0L);
    }

    public static Integer nullToIntegerZero(@Nullable Long val) {
        return nvl(val, 0).intValue();
    }

    public static BigDecimal nullToZero(@Nullable BigDecimal val) {
        return nvl(val, BigDecimal.ZERO);
    }

    public static Integer zeroToNull(@Nullable Integer val) {
        return val == null || val == 0 ? null : val;
    }

    public static Long zeroToNull(@Nullable Long val) {
        return val == null || val == 0 ? null : val;
    }

}
