package ru.yandex.direct.common.util;

import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.utils.text.StringModifier;

/**
 * Вспомогательный класс для работы с текстами
 */
public class TextUtils {
    private static final StringModifier SPACE_CLEAN_MODIFIER;
    private static final StringModifier SMART_STRIP_MODIFIER;
    private static final StringModifier SMART_STRIP_WITH_ANGLE_QUOTES_MODIFIER;

    private TextUtils() {
    }

    static {
        SPACE_CLEAN_MODIFIER = new StringModifier.Builder()
                // Удаляем концевые пробелы и заменяем несколько последовательных пробельных символов на один пробел
                .withCustomRule(StringUtils::normalizeSpace)
                // Удаляем пробельные символы перед знаками пунктуации
                .withRegexpReplaceAllRule(
                        Pattern.compile("(?<=[\\w])\\s+(?=[.,;!:()?])", Pattern.UNICODE_CHARACTER_CLASS), "")
                // Удаляем пробельные символы после знаками пунктуации
                .withRegexpReplaceAllRule(
                        Pattern.compile("(?<=[.,;!:()?])\\s+(?=[\\w])", Pattern.UNICODE_CHARACTER_CLASS), "")
                .build();

        StringModifier.Builder builder = new StringModifier.Builder()
                // Удаляем концевые пробелы и заменяем несколько последовательных пробельных символов на один пробел
                .withCustomRule(StringUtils::normalizeSpace)
                // Заменяем разные разновидности дефисов на -
                .withRegexpReplaceAllRule("[\u2010\u2011\u2012\u2015\u0096]", "-")
                // Заменяем различные разновидности двойных кавычек на "
                .withRegexpReplaceAllRule("[\u203a\u2039\u2033\u201e\u201d\u201c\u201a\u201b\u2018]", "\"")
                // Заменяем различные разновидности одинарных кавычек на '
                .withRegexpReplaceAllRule("[\u2032`]", "'");
        SMART_STRIP_MODIFIER = builder.build();

        // Заменяем угловые кавычки на обычные
        builder.withRegexpReplaceAllRule("[\u00ab\u00bb]", "\"");
        SMART_STRIP_WITH_ANGLE_QUOTES_MODIFIER = builder.build();
    }

    /**
     * Удалить концевые пробелы, заменить несколько пробелов на один, заменить различные дефисы на обычный,
     * заменить различные одинарные и двойные кавычки на обычные
     * <p>
     * Равносильно:
     * <code>smartStrip(text, false)</code>
     */
    public static String smartStrip(String text) {
        return smartStrip(text, false);
    }

    /**
     * Удалить концевые пробелы, заменить несколько пробелов на один, заменить различные дефисы на обычный,
     * заменить различные одинарные и двойные кавычки на обычные
     *
     * @return Преобразованный текст
     */
    public static String smartStrip(String text, boolean replaceAngleQuotes) {
        if (!replaceAngleQuotes) {
            return SMART_STRIP_MODIFIER.makeReplacements(text);
        } else {
            return SMART_STRIP_WITH_ANGLE_QUOTES_MODIFIER.makeReplacements(text);
        }
    }

    /**
     * 1. Удаляет пробельние символы в начале/конце строки.
     * 2. Заменяет множественные пробелы одним.
     * 3. Удаляет пробелы между буквами и пунктуацией.
     */
    public static String spaceCleaner(String text) {
        return SPACE_CLEAN_MODIFIER.makeReplacements(text);
    }

}
