package ru.yandex.direct.core;

import java.util.Locale;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.Translator;

@ParametersAreNonnullByDefault
public abstract class TranslatableException extends RuntimeException {

    private static final Translator TRANSLATOR = I18NBundle.makeStubTranslatorFactory().getTranslator(Locale.ENGLISH);

    private int code;
    private Translatable shortMessage;
    private Translatable detailedMessage;

    public TranslatableException(String internalDescription, Throwable cause, int code,
                                 Translatable shortMessage, @Nullable Translatable detailedMessage) {
        super(toMessage(internalDescription, shortMessage, detailedMessage), cause);
        populate(code, shortMessage, detailedMessage);
    }

    public TranslatableException(String internalDescription, int code,
                                 Translatable shortMessage, @Nullable Translatable detailedMessage) {
        super(toMessage(internalDescription, shortMessage, detailedMessage));
        populate(code, shortMessage, detailedMessage);
    }

    public TranslatableException(Throwable cause, int code,
                                 Translatable shortMessage, @Nullable Translatable detailedMessage) {
        super(toMessage(null, shortMessage, detailedMessage), cause);
        populate(code, shortMessage, detailedMessage);
    }

    public TranslatableException(int code, Translatable message) {
        super(toMessage(null, message, null));
        populate(code, message, null);
    }

    public TranslatableException(int code, Translatable shortMessage, @Nullable Translatable detailedMessage) {
        super(toMessage(null, shortMessage, detailedMessage));
        populate(code, shortMessage, detailedMessage);
    }

    private static String toMessage(@Nullable String internalDescription, Translatable shortMessage,
                                    @Nullable Translatable detailedMessage) {
        StringBuilder sb = new StringBuilder();
        if (internalDescription != null) {
            sb.append(internalDescription).append(". ");
        }
        sb.append("short message: ").append(shortMessage.translate(TRANSLATOR));
        if (detailedMessage != null) {
            sb.append("; details: ").append(detailedMessage.translate(TRANSLATOR));
        }
        return sb.toString();
    }

    public int getCode() {
        return code;
    }

    public Translatable getShortMessage() {
        return shortMessage;
    }

    @Nullable
    public Translatable getDetailedMessage() {
        return detailedMessage;
    }

    private void populate(int code, Translatable shortMessage, @Nullable Translatable detailedMessage) {
        this.code = code;
        this.shortMessage = shortMessage;
        this.detailedMessage = detailedMessage;
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{" +
                "code=" + code +
                ", shortMessage=" + shortMessage +
                ", detailedMessage=" + detailedMessage +
                '}';
    }
}
