package ru.yandex.direct.core.aggregatedstatuses;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesCampaignRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesClientRepository;
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaignMoney;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusWallet;
import ru.yandex.direct.core.entity.campaign.aggrstatus.WalletStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.utils.CommonUtils;

import static ru.yandex.direct.dbutil.model.ClientId.fromLong;

@Service
@ParametersAreNonnullByDefault
public class AggregatedStatusesCampaignService {
    private final Logger logger = LoggerFactory.getLogger(AggregatedStatusesCampaignService.class);

    private final AggregatedStatusesClientRepository clientRepository;
    private final AggregatedStatusesCampaignRepository campaignRepository;

    @Autowired
    public AggregatedStatusesCampaignService(AggregatedStatusesClientRepository clientRepository,
                                             AggregatedStatusesCampaignRepository campaignRepository) {
        this.clientRepository = clientRepository;
        this.campaignRepository = campaignRepository;
    }

    public List<AggregatedStatusCampaign> getCampaigns(int shard, Collection<Long> campaignIds) {
        return campaignRepository.getCampaigns(shard, campaignIds);
    }

    public Map<Long, AggregatedStatusCampaign> getCampaignById(int shard, Collection<Long> campaignIds) {
        return campaignRepository.getCampaignById(shard, campaignIds);
    }

    public List<AggregatedStatusWallet> getWallets(int shard, Collection<Long> walletIds) {
        List<AggregatedStatusWallet> wallets = new ArrayList<>();
        List<AggregatedStatusCampaign> campaignsTypeWallet = campaignRepository.getWallets(shard, walletIds);
        Map<Long, List<AggregatedStatusCampaignMoney>> campaignsByWalletId =
                campaignRepository.getCampaignsMoneyByWalletIds(shard, walletIds);

        Map<Long, Client> clientsById = clientRepository.getClientsById(shard,
                campaignsTypeWallet.stream().map(c -> fromLong(c.getClientId())).collect(Collectors.toSet()));

        for (AggregatedStatusCampaign wc : campaignsTypeWallet) {
            if (!clientsById.containsKey(wc.getClientId())) {
                logger.error("For wallet with cid: {} wasn't found client with id: {}. Skipping wallet",
                        wc.getWalletId(), wc.getClientId());
                continue;
            }
            AggregatedStatusWallet wallet = campaignToWallet(
                    wc,
                    campaignsByWalletId.getOrDefault(wc.getId(), Collections.emptyList()),
                    clientsById.get(wc.getClientId())
            );
            wallets.add(wallet);
        }
        return wallets;
    }

    public Map<Long, List<Long>> getSubCampaignIdsByMasterId(int shard, Collection<Long> campaignIds) {
        return campaignRepository.getSubCampaignIdsByMasterId(shard, campaignIds);
    }

    public Set<Long> getCampaignIdsByWalletIds(int shard, Collection<Long> walletIds) {
        Set<Long> filteredIds = walletIds.stream().filter(CommonUtils::isValidId).collect(Collectors.toSet());
        return campaignRepository.getCampaignIdsByWalletIds(shard, filteredIds);
    }

    private AggregatedStatusWallet campaignToWallet(AggregatedStatusCampaign wallet,
                                                    List<AggregatedStatusCampaignMoney> campaignsUnderWallet,
                                                    Client client) {
        Money moneyOnCamps = Money.valueOf(BigDecimal.ZERO, wallet.getCurrencyCode());
        Money walletSumDebt = Money.valueOf(BigDecimal.ZERO, wallet.getCurrencyCode());
        for (var campaign : campaignsUnderWallet) {
            if (!wallet.getCurrencyCode().equals(campaign.getCurrencyCode())) {
                continue;
            }
            Money leftOnCampaign =
                    Money.valueOf(campaign.getSum().subtract(campaign.getSumSpent()), campaign.getCurrencyCode());
            if (leftOnCampaign.greaterThanZero()) {
                moneyOnCamps = moneyOnCamps.add(leftOnCampaign);
            } else {
                walletSumDebt = walletSumDebt.add(leftOnCampaign);
            }
        }
        Money moneyOnWallet = Money.valueOf(wallet.getSum(), wallet.getCurrencyCode()).add(walletSumDebt);

        BigDecimal autoOverdraftAddition = AutoOverdraftUtils.calculateAutoOverdraftAddition(wallet.getCurrencyCode(),
                wallet.getSum(), walletSumDebt.bigDecimalValue(), toClientAutoOverdraftInfo(client));

        Boolean walletEnabled = client.getCanPayBeforeModeration() || !campaignsUnderWallet.isEmpty();

        return new AggregatedStatusWallet()
                .withId(wallet.getId())
                .withCurrency(wallet.getCurrencyCode())
                .withSum(moneyOnWallet.roundToCentDown().bigDecimalValue())
                .withAutoOverdraftAddition(Money.valueOf(autoOverdraftAddition, wallet.getCurrencyCode())
                        .roundToCentDown().bigDecimalValue())
                .withStatus(getWalletStatus(wallet, walletEnabled));
    }

    private static WalletStatus getWalletStatus(AggregatedStatusCampaign wallet, Boolean enabled) {
        boolean campaignWaitingForPayment
                = isCampaignWaitingForPayment(wallet, calcCampaignEffectiveModerationStatus(wallet));
        return new WalletStatus()
                .withEnabled(enabled)
                .withWaitingForPayment(campaignWaitingForPayment)
                .withBudgetLimitationStopTime(getWalletBudgetStopTime(wallet));
    }

    private static boolean isCampaignWaitingForPayment(AggregatedStatusCampaign campaign,
                                        CampaignStatusModerate effectiveModerationStatus) {
        return !campaign.getArchived() && campaign.getSumToPay().compareTo(Currencies.EPSILON) > 0 &&
                effectiveModerationStatus != CampaignStatusModerate.NO;
    }

    public static CampaignStatusModerate calcCampaignEffectiveModerationStatus(AggregatedStatusCampaign campaign) {
        if (campaign.getStatusModerate() == CampaignStatusModerate.YES &&
                campaign.getStatusPostModerate() == CampaignStatusPostmoderate.NO) {
            return CampaignStatusModerate.NO;
        }
        return campaign.getStatusModerate();
    }

    private static LocalDateTime getWalletBudgetStopTime(AggregatedStatusCampaign wallet) {
        return wallet.getDayBudget() != null &&
               wallet.getDayBudget().compareTo(BigDecimal.ZERO) > 0 &&
               wallet.getDayBudgetStopTime() != null
               ?  wallet.getDayBudgetStopTime() : null;
    }

    public static ClientAutoOverdraftInfo toClientAutoOverdraftInfo(Client client) {
        return new ClientAutoOverdraftInfo()
                .withClientId(client.getId())
                .withDebt(client.getDebt())
                .withStatusBalanceBanned(client.getStatusBalanceBanned())
                .withOverdraftLimit(client.getOverdraftLimit())
                .withAutoOverdraftLimit(client.getAutoOverdraftLimit());
    }
}
