package ru.yandex.direct.core.aggregatedstatuses;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.aggregatedstatuses.grut.GrutAggregatedStatusService;
import ru.yandex.direct.core.aggregatedstatuses.logic.AdGroupStates;
import ru.yandex.direct.core.aggregatedstatuses.logic.CampaignStates;
import ru.yandex.direct.core.aggregatedstatuses.logic.KeywordStates;
import ru.yandex.direct.core.aggregatedstatuses.logic.RetargetingStates;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesAdGroupRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesArchivedKeywordRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesBannerRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesKeywordRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.BannerDataForStatusService;
import ru.yandex.direct.core.aggregatedstatuses.repository.ShowConditionsCounter;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RejectReasonsAndCountersByIds;
import ru.yandex.direct.core.aggregatedstatuses.service.AggregatedStatusesHelper;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.KeywordStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.RetargetingStatesEnum;
import ru.yandex.direct.core.entity.autobudget.service.CpaAutobudgetPessimizedUsersService;
import ru.yandex.direct.core.entity.banner.type.banneradditions.BannerAdditionsRepository;
import ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeRepository;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusWallet;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.aggrstatus.StatusAggregationKeyword;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.promoextension.PromoExtensionRepository;
import ru.yandex.direct.core.entity.promoextension.model.PromoExtension;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.GoalConversion;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.CommonUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGGREGATED_STATUS_PARALLEL;
import static ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_CALCULATION_LACK_OF_CONVERSION_AGGREGATED_STATUS;
import static ru.yandex.direct.common.db.PpcPropertyNames.FETCH_DATA_FROM_GRUT_FOR_AGGR_STATUSES;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_NEW_SELF_STATUS_CALCULATOR;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.adChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.adgroupChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.bidBaseChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.campaignChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.retargetingChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.logic.SelfStatusCalculators.calcAdGroupSelfStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.SelfStatusCalculators.calcCampaignSelfStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.SelfStatusCalculators.calcKeywordSelfStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.SelfStatusCalculators.calcRelevanceMatchSelfStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.SelfStatusCalculators.calcRetargetingSelfStatus;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/*
 * Сервис для вычисления статусов объектов из записи этих статусов БД.
 * Подробнее о системе статусов см. документацию: https://docs.yandex-team.ru/direct-dev/aggr-statuses/concept.html
 */
@Service
@ParametersAreNonnullByDefault
public class AggregatedStatusesService {
    // Подобъекты поднимаются только по id-шникам, но без чанкования
    private static final int MAX_CAMPAIGN_IDS_TO_FULLY_RECOUNT = 10000;
    // для кампаний и групп цифры маленькие потому поднимаем подобъекты
    // 1000 - групп на кампанию, 200 фраз на группу (но кампании редко забиты под завязку, а группы часто)
    private static final int CIDS_CHUNK_SIZE = 500;
    private static final int PIDS_CHUNK_SIZE = 500;
    private static final int BIDS_CHUNK_SIZE = 10000;
    private static final int KWIDS_CHUNK_SIZE = 10000;
    private static final int ARC_KWIDS_CHUNK_SIZE = 1000; // DIRECT-109906
    private static final int CALLOUTS_CHUNK_SIZE = 1000; // на случай если один привязан к многим объявлениям
    private static final int PERF_CREATIVES_CHUNK_SIZE = 10000; // на случай если один привязан к многим объявлениям
    private static final int RETIDS_CHUNK_SIZE = 1000;
    private static final Logger logger = LoggerFactory.getLogger(AggregatedStatusesService.class);

    private final AggregatedStatusesHelper aggregatedStatusesHelper;
    private final AggregatedStatusesBannerRepository bannerRepository;
    private final AggregatedStatusesCampaignService campaignService;
    private final CpaAutobudgetPessimizedUsersService pessimizedLoginsService;
    private final AggregatedStatusesAdGroupRepository adGroupRepository;
    private final AdGroupService adGroupService;
    private final ClientService clientService;
    private final ClientNdsService clientNdsService;
    private final AggregatedStatusesKeywordRepository keywordRepository;
    private final AggregatedStatusesArchivedKeywordRepository archivedKeywordRepository;
    private final BidRepository bidRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final BannerAdditionsRepository bannerAdditionsRepository;
    private final BannerCreativeRepository newBannerCreativeRepository;
    private final RetargetingRepository retargetingRepository;
    private final PromoExtensionRepository promoExtensionRepository;
    private final PpcProperty<Boolean> aggregatedStatusParallelProperty;
    private final KeywordStates keywordStates = new KeywordStates();
    private final AdGroupStates adGroupStates = new AdGroupStates();
    private final CampaignStates campaignStates = new CampaignStates();
    private final RetargetingStates retargetingStates = new RetargetingStates();
    private final PpcProperty<Boolean> lackOfConversionStatusEnabled;
    private final PpcProperty<Boolean> useNewSelfStatusCalculator;
    private final PpcProperty<Boolean> fetchDataFromGrutForAggrStatuses;
    private final BannerDataForStatusService bannerDataForStatusService;
    private final GrutAggregatedStatusService grutAggregatedStatusService;
    private final AdChangesProcessor adChangesProcessor = new AdChangesProcessor();

    @Autowired
    public AggregatedStatusesService(AggregatedStatusesHelper aggregatedStatusesHelper,
                                     AggregatedStatusesBannerRepository bannerRepository,
                                     AggregatedStatusesAdGroupRepository adGroupRepository,
                                     AdGroupService adGroupService,
                                     AggregatedStatusesCampaignService campaignService,
                                     CpaAutobudgetPessimizedUsersService pessimizedLoginsService,
                                     ClientService clientService, ClientNdsService clientNdsService,
                                     AggregatedStatusesKeywordRepository keywordRepository,
                                     AggregatedStatusesArchivedKeywordRepository archivedKeywordRepository,
                                     BidRepository bidRepository,
                                     AggregatedStatusesRepository aggregatedStatusesRepository,
                                     BannerAdditionsRepository bannerAdditionsRepository,
                                     BannerCreativeRepository newBannerCreativeRepository,
                                     RetargetingRepository retargetingRepository,
                                     PromoExtensionRepository promoExtensionRepository,
                                     BannerDataForStatusService bannerDataForStatusService,
                                     PpcPropertiesSupport ppcPropertiesSupport,
                                     GrutAggregatedStatusService grutAggregatedStatusService) {
        this.aggregatedStatusesHelper = aggregatedStatusesHelper;
        this.bannerRepository = bannerRepository;
        this.adGroupRepository = adGroupRepository;
        this.adGroupService = adGroupService;
        this.campaignService = campaignService;
        this.pessimizedLoginsService = pessimizedLoginsService;
        this.clientService = clientService;
        this.clientNdsService = clientNdsService;
        this.keywordRepository = keywordRepository;
        this.archivedKeywordRepository = archivedKeywordRepository;
        this.bidRepository = bidRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.bannerAdditionsRepository = bannerAdditionsRepository;
        this.newBannerCreativeRepository = newBannerCreativeRepository;
        this.retargetingRepository = retargetingRepository;
        this.promoExtensionRepository = promoExtensionRepository;
        this.aggregatedStatusParallelProperty = ppcPropertiesSupport.get(
                AGGREGATED_STATUS_PARALLEL,
                Duration.ofMinutes(1)
        );
        this.lackOfConversionStatusEnabled = ppcPropertiesSupport.get(
                ENABLE_CALCULATION_LACK_OF_CONVERSION_AGGREGATED_STATUS, Duration.ofMinutes(1));
        this.useNewSelfStatusCalculator = ppcPropertiesSupport.get(
                USE_NEW_SELF_STATUS_CALCULATOR, Duration.ofMinutes(1));
        this.fetchDataFromGrutForAggrStatuses = ppcPropertiesSupport.get(
                FETCH_DATA_FROM_GRUT_FOR_AGGR_STATUSES, Duration.ofMinutes(5));

        this.bannerDataForStatusService = bannerDataForStatusService;
        this.grutAggregatedStatusService = grutAggregatedStatusService;
    }

    /*
        В одной из тестовых выборок на 10 кампаний было
            3895    - групп
            13327   - объявлений
            19174   - фраз
     */
    public void fullyRecalculateStatuses(int shard, LocalDateTime updateBefore, Set<Long> campaignIds,
                                         RecalculationDepthEnum recalculationDepth) {
        if (campaignIds.size() > MAX_CAMPAIGN_IDS_TO_FULLY_RECOUNT) {
            throw new IllegalArgumentException("Campaign ids list can't be longer than "
                    + MAX_CAMPAIGN_IDS_TO_FULLY_RECOUNT + ", please partition it outside call");
        }

        switch (recalculationDepth) {
            case CAMPAIGNS:
                recalculateCampaignStatuses(shard, updateBefore, campaignIds);
                break;

            case ADGROUPS:
                recalculateAdGroupStatuses(shard, updateBefore, campaignIds);
                break;

            case ADGROUPSUBJECTS:
                recalculateAdGroupSubjectStatuses(shard, updateBefore, campaignIds);
                break;

            case ALL:
            default:
                recalculateAllStatuses(shard, updateBefore, campaignIds);
                break;
        }
    }

    private void recalculateCampaignStatuses(int shard, LocalDateTime updateBefore, Set<Long> campaignIds) {
        var changes = new ChangesHolder(
                campaignIds.stream().map(c -> campaignChangesEvent(c, false))
                        .collect(Collectors.toUnmodifiableList()),
                List.of(), List.of(), List.of(), List.of(), List.of(), List.of(), List.of(), List.of(), List.of());
        processChanges(shard, updateBefore, changes, RecalculationDepthEnum.CAMPAIGNS);
    }

    private void recalculateAdGroupStatuses(int shard, LocalDateTime updateBefore, Set<Long> campaignIds) {
        var adGroupsSimple = adGroupRepository.getAdGroupsByCampaignIds(shard, campaignIds);
        var changes = new ChangesHolder(
                List.of(),
                adGroupsSimple.stream().map(g -> adgroupChangesEvent(g.getId(), g.getCampaignId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                List.of(), List.of(), List.of(), List.of(), List.of(), List.of(), List.of(), List.of());
        processChanges(shard, updateBefore, changes, RecalculationDepthEnum.ADGROUPS);
    }

    private void recalculateAdGroupSubjectStatuses(int shard, LocalDateTime updateBefore,
                                                   Set<Long> campaignIds) {
        var adGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds);

        // sorted and unique
        var bids = bannerRepository.getBannerIdsByAdGroupIds(shard, adGroupIds);
        var ads = chunked(bids, BIDS_CHUNK_SIZE).toFlatList(bidsChunk -> bannerRepository.getBanners(shard, bidsChunk));

        // сразу эвенты, не красиво, но этих объектов много и экономия оправдана
        var keywordEvents = keywordRepository.getKeywordChangeEventsByAdGroupIds(shard, new HashSet<>(adGroupIds));

        var bidsBase = bidRepository.getRelevanceMatchByCampaignIdsNotDeleted(shard, new ArrayList<>(campaignIds));

        var retargetings = retargetingRepository.getRetargetingsByAdGroups(shard, adGroupIds);

        var changes = new ChangesHolder(
                List.of(),
                List.of(),
                ads.stream().map(b -> adChangesEvent(b.getId(), b.getAdgroupId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                keywordEvents,
                bidsBase.stream().map(b -> bidBaseChangesEvent(b.getId(), b.getAdGroupId()))
                        .collect(Collectors.toUnmodifiableList()),
                List.of(),
                List.of(),
                List.of(),
                retargetings.stream().map(r -> retargetingChangesEvent(r.getId(), r.getAdGroupId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                List.of());
        processChanges(shard, updateBefore, changes, RecalculationDepthEnum.ADGROUPSUBJECTS);
    }

    private void recalculateAllStatuses(int shard, LocalDateTime updateBefore, Set<Long> campaignIds) {
        var adgroups = adGroupRepository.getAdGroupsByCampaignIds(shard, campaignIds);

        var adGroupIds = adgroups.stream().map(AggregatedStatusAdGroup::getId).collect(Collectors.toSet());

        // sorted and unique
        var bids = bannerRepository.getBannerIdsByAdGroupIds(shard, adGroupIds);
        var ads = chunked(bids, BIDS_CHUNK_SIZE).toFlatList(bidsChunk -> bannerRepository.getBanners(shard, bidsChunk));

        // сразу эвенты, не красиво, но этих объектов много и экономия оправдана
        var keywordEvents = keywordRepository.getKeywordChangeEventsByAdGroupIds(shard, adGroupIds);

        var bidsBase = bidRepository.getRelevanceMatchByCampaignIdsNotDeleted(shard, new ArrayList<>(campaignIds));

        var retargetings = retargetingRepository.getRetargetingsByAdGroups(shard, adGroupIds);

        var changes = new ChangesHolder(
                campaignIds.stream().map(c -> campaignChangesEvent(c, false))
                        .collect(Collectors.toUnmodifiableList()),
                adgroups.stream().map(g -> adgroupChangesEvent(g.getId(), g.getCampaignId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                ads.stream().map(b -> adChangesEvent(b.getId(), b.getAdgroupId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                keywordEvents,

                bidsBase.stream().map(b -> bidBaseChangesEvent(b.getId(), b.getAdGroupId()))
                        .collect(Collectors.toUnmodifiableList()),
                List.of(),
                List.of(),
                List.of(),
                retargetings.stream().map(r -> retargetingChangesEvent(r.getId(), r.getAdGroupId(), false))
                        .collect(Collectors.toUnmodifiableList()),
                List.of());
        processChanges(shard, updateBefore, changes, RecalculationDepthEnum.ALL);
    }

    public void processChanges(int shard, @Nullable LocalDateTime updateBefore, ChangesHolder changes,
                               RecalculationDepthEnum recalculationDepth) {
        new OnShardChangesProcessor(shard,
                recalculationDepth,
                updateBefore,
                this.useNewSelfStatusCalculator.getOrDefault(false)
        ).processChanges(changes);
    }

    private <T> StreamEx<List<T>> chunked(Iterable<T> iterable, int size) {
        return StreamEx.of(Iterables.partition(iterable, size).iterator());
    }

    @SafeVarargs
    private <K> Set<K> mergeSets(Set<K>... sourceSets) {
        Set<K> result = new HashSet<>();
        for (Set<K> sourceSet : sourceSets) {
            result.addAll(sourceSet);
        }
        return result;
    }

    // set2 - хотим Set с дешевым contains, иначе получаем квадрат по производительности
    // (abc, bcd) => bc
    private <K> Set<K> idsIntersection(Collection<K> set1, Set<K> set2) {
        Set<K> result = new HashSet<>(set1);
        result.retainAll(set2);
        return result;
    }

    // subtraction - хотим Set с дешевым contains, иначе получаем квадрат по производительности
    // (abc, bcd) => a
    private <K> Set<K> subtractIds(Collection<K> subtractFrom, Set<K> subtraction) {
        HashSet<K> result = new HashSet<>(subtractFrom);
        result.removeAll(subtraction);
        return result;
    }

    class OnShardChangesProcessor { // обертка, чтобы шард везде не таскать
        private final int shard;

        private final RecalculationDepthEnum recalculationDepth;

        @Nullable
        private final LocalDateTime updateBefore;

        private final boolean useNewSelfStatusCalculator;

        OnShardChangesProcessor(int shard,
                                RecalculationDepthEnum recalculationDepth,
                                @Nullable LocalDateTime updateBefore,
                                boolean useNewSelfStatusCalculator) {
            this.shard = shard;
            this.recalculationDepth = recalculationDepth;
            this.updateBefore = updateBefore;
            this.useNewSelfStatusCalculator = useNewSelfStatusCalculator;
        }

        /**
         * Метод для пересчета статусов сущностей определенного уровня (для таски по пересчету).
         *
         * @param changes Список сущностей, требующих пересчета.
         */
        void processChanges(ChangesHolder changes) {
            logger.debug("Processing changes in {}", changes);
            switch (recalculationDepth) {
                case ALL:
                    processAllChanges(changes);
                    break;

                case CAMPAIGNS:
                    processChangedCampaigns(changes.getCampaignsChanges(), Set.of(), Set.of());
                    break;

                case ADGROUPS:
                    processChangesAdGroups(changes.getAdgroupsChanges(), Set.of(), Set.of(), Set.of(), Set.of());
                    break;

                case ADGROUPSUBJECTS:
                    processChangesAds(changes.getAdsChanges(), Set.of(), Set.of());
                    processChangesKeywords(changes.getKeywordsChanges(), changes.getBidsBaseChanges());
                    processChangesRetargetings(changes.getRetargetingsChanges());
                    break;
            }
        }

        /**
         * Метод для "органического" пересчета агрегированных статусов.
         *
         * @param changes Список сущностей, требующих пересчета.
         */
        private void processAllChanges(ChangesHolder changes) {
            processChangedCampaigns(changes.getCampaignsChanges(),
                    processChangesAdGroups(changes.getAdgroupsChanges(),
                            processChangesAds(changes.getAdsChanges(),
                                    processChangesCallouts(changes.getCalloutsChanges()),
                                    processChangesPerfCreatives(changes.getPerfCreativesChanges())),
                            processChangesKeywords(changes.getKeywordsChanges(), changes.getBidsBaseChanges()),
                            processRestrictedGeoChanges(changes.getRestrictedGeoChanges()),
                            processChangesRetargetings(changes.getRetargetingsChanges())),
                    processChangesPromoExtensions(changes.getPromoExtensionChanges()));
        }

        private void processChangedCampaigns(Collection<ChangesHolder.CampaignChangesEvent> campaignEvents,
                                             Set<Long> changedAdGroupsCids,
                                             Set<Long> changedPromoExtensionsCids) {
            if (campaignEvents.isEmpty() && changedAdGroupsCids.isEmpty() && changedPromoExtensionsCids.isEmpty()) {
                return;
            }
            boolean parallel = aggregatedStatusParallelProperty.getOrDefault(false);
            var isCampaignsOnlyRecalculation = recalculationDepth == RecalculationDepthEnum.CAMPAIGNS;
            Set<Long> cids = campaignEvents.stream()
                    .filter(not(ChangesHolder.CampaignChangesEvent::isDeleted))
                    .map(ChangesHolder.CampaignChangesEvent::getCampaignId).collect(Collectors.toSet());
            Set<Long> cidsOnChangedWallets = cidsAffectedByWalletChanges(cids);
            // сортировка немного улучшит локальность данных и ускорит запросы
            List<Long> allCids = mergeSets(cids, changedAdGroupsCids, cidsOnChangedWallets, changedPromoExtensionsCids)
                    .stream().sorted().collect(Collectors.toList());
            try (TraceProfile ignore =
                         Trace.current().profile("aggr_statuses:process_campaigns", "", allCids.size())
            ) {
                // HINT: можно распараллелить
                for (List<Long> cidsChunk : Iterables.partition(allCids, CIDS_CHUNK_SIZE)) {
                    Set<Long> cidsToRecountChunk = isCampaignsOnlyRecalculation
                            ? new HashSet<>(cidsChunk)
                            : idsIntersection(cidsChunk, changedAdGroupsCids);
                    var reasonsAndCounters = recountCampaignAdGroups(cidsToRecountChunk, parallel);
                    aggregateOnCampaignsChange(cidsChunk, reasonsAndCounters.getRejectReasons(),
                            reasonsAndCounters.getCounters());
                }
            }
            Set<Long> deletedCids = campaignEvents.stream()
                    .filter(ChangesHolder.CampaignChangesEvent::isDeleted)
                    .map(ChangesHolder.CampaignChangesEvent::getCampaignId)
                    .collect(Collectors.toSet());
            try (TraceProfile ignore =
                         Trace.current()
                                 .profile("aggr_statuses:process_campaigns_delete", "", deletedCids.size())
            ) {
                aggregatedStatusesRepository.deleteCampaignStatuses(shard, deletedCids);
            }
        }

        /*  часть cid-ов, по которым пришли события, может быть кошельками, при изменении информации на кошельке надо
            пересчитать кампании под ним, для чего надо собрать их id. Данный метод по переданным id кошельков
            возвращает id кампаний под ними, можно передавать id обычных кампаний, они будут проигнорированы
         */
        private Set<Long> cidsAffectedByWalletChanges(Set<Long> cids) {
            return campaignService.getCampaignIdsByWalletIds(shard, cids);
        }

        private Set<Long> cidsAffectedByPromoExtensionChanges(Set<Long> promoIds) {
            return flatMapToSet(
                    promoExtensionRepository.getCidsByPromoExtensionIds(promoIds).values(), Function.identity());
        }

        private RejectReasonsAndCountersByIds<CampaignCounters> recountCampaignAdGroups(@Nullable Set<Long> cids,
                                                                                        boolean parallel) {
            Map<Long, Map<ModerationDiagType, Set<Long>>> reasonsByCid = new HashMap<>();
            Map<Long, CampaignCounters> countersByCid = new HashMap<>();
            Map<Long, List<AggregatedStatusAdGroupData>> adGroupsStatusesByCid =
                    aggregatedStatusesRepository.getAdGroupStatusesOnCampaignsForCount(shard, cids, parallel);
            for (Long cid : adGroupsStatusesByCid.keySet()) {
                var reasons = reasonsByCid.computeIfAbsent(cid, c -> new EnumMap<>(ModerationDiagType.class));
                var counters = countersByCid.computeIfAbsent(cid, c -> new CampaignCounters());
                for (var statusData : adGroupsStatusesByCid.getOrDefault(cid, emptyList())) {
                    for (var entry : statusData.getRejectReasons().entrySet()) {
                        reasons.computeIfAbsent(entry.getKey(), k -> new HashSet<>()).addAll(entry.getValue());
                    }
                    counters.incrAdgroupsCount();
                    if (statusData.getStatus().isPresent()) {
                        counters.incr(statusData.getStates());
                        counters.incr(statusData.getStatus().get());
                    }
                }
            }
            return new RejectReasonsAndCountersByIds<>(reasonsByCid, countersByCid);
        }

        private Set<Long> processChangesAdGroups(Collection<ChangesHolder.AdgroupChangesEvent> adgroupEvents,
                                                 Set<Long> changedAdsPids,
                                                 Set<Long> changedKeywordsPids,
                                                 Set<Long> changedRestrictedGeoPids,
                                                 Set<Long> changedRetargetingsPids) {
            /* Берем id групп, которые поменялись сами, берем id групп у которых поменялись объявления и фразы,
               мерджим эти списки, результирующий список бьем на чанки. Для каждого чанка подмножество
               pidsToRecountChunk - группы у которых поменялись субъекты (получаем пересечением) Все группы чанка нужно,
               переагрегировать, счетчики для тех групп, чьи счетчики мы не пересчитывали берем из базы. */
            if (adgroupEvents.isEmpty() && changedAdsPids.isEmpty() && changedKeywordsPids.isEmpty()
                    && changedRestrictedGeoPids.isEmpty() && changedRetargetingsPids.isEmpty()) {
                return Set.of();
            }
            var isAdGroupsOnlyRecalculation = recalculationDepth == RecalculationDepthEnum.ADGROUPS;
            Set<Long> pids = adgroupEvents.stream()
                    .filter(not(ChangesHolder.AdgroupChangesEvent::isDeleted))
                    .map(ChangesHolder.AdgroupChangesEvent::getAdgroupId).collect(Collectors.toSet());
            Set<Long> subjectsChangedPids = mergeSets(changedAdsPids, changedKeywordsPids, changedRetargetingsPids);
            // сортировка немного улучшит локальность данных и ускорит запросы
            List<Long> allPids = mergeSets(pids, subjectsChangedPids, changedRestrictedGeoPids)
                    .stream().sorted().collect(Collectors.toList());
            Set<Long> affectedCids = new HashSet<>();
            try (TraceProfile ignore =
                         Trace.current().profile("aggr_statuses:process_adgroups", "", allPids.size())
            ) {
                for (List<Long> pidsChunk : Iterables.partition(allPids, PIDS_CHUNK_SIZE)) {
                    // пересчитывает счетчики но только у групп из текущего чанка
                    Set<Long> pidsToRecountChunk = isAdGroupsOnlyRecalculation
                            ? new HashSet<>(pidsChunk)
                            : idsIntersection(pidsChunk, subjectsChangedPids);
                    var reasonsAndCounters = recountAdGroupsAdsKeywordsAndRetargetings(pidsToRecountChunk);
                    aggregateOnAdGroupsChange(pidsChunk, reasonsAndCounters.getRejectReasons(),
                            reasonsAndCounters.getCounters());
                    if (!isAdGroupsOnlyRecalculation) {
                        affectedCids.addAll(
                                adGroupRepository.getCampaignIdsByAdGroupIds(shard, pidsChunk));
                    }
                }
            }
            Set<Long> deletedAdGroupIds = adgroupEvents.stream()
                    .filter(ChangesHolder.AdgroupChangesEvent::isDeleted)
                    .map(ChangesHolder.AdgroupChangesEvent::getAdgroupId)
                    .collect(Collectors.toSet());
            try (TraceProfile ignore =
                         Trace.current().profile("aggr_statuses:process_adgroups_delete", "", allPids.size())
            ) {
                aggregatedStatusesRepository.deleteAdGroupStatuses(shard, deletedAdGroupIds);
            }
            return isAdGroupsOnlyRecalculation
                    ? Set.of()
                    : addCidsFromAdgroupChangeEvents(affectedCids, adgroupEvents);
        }

        // там где cid заполнен берем его их события. Для событий из второстепенных таблиц, где cid-а нет
        // ищем его по pid-у в БД. Нужны оба способа:
        //  * первый не работает для второстепенных таблиц (напр. adgroups_cpm_banner), где нет cid-а
        //  * второго при удалении из основной таблицы (phrases), т.к. неоткуда достать cid
        private Set<Long> addCidsFromAdgroupChangeEvents(Set<Long> affectedCids,
                                                         Collection<ChangesHolder.AdgroupChangesEvent> adgroupEvents) {
            if (adgroupEvents.isEmpty()) {
                return affectedCids;
            }
            Set<Long> eventFilledCids = adgroupEvents.stream()
                    .map(ChangesHolder.AdgroupChangesEvent::getCampaignId)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());
            Set<Long> pidsWithMissingCids = adgroupEvents.stream()
                    .filter(g -> g.getCampaignId() == null)
                    .map(ChangesHolder.AdgroupChangesEvent::getAdgroupId)
                    .collect(Collectors.toSet());
            affectedCids.addAll(eventFilledCids);
            if (!pidsWithMissingCids.isEmpty()) {
                affectedCids.addAll(adGroupRepository.getCampaignIdsByAdGroupIds(shard, pidsWithMissingCids));
            }
            return affectedCids;
        }

        private RejectReasonsAndCountersByIds<AdGroupCounters> recountAdGroupsAdsKeywordsAndRetargetings(@Nullable Set<Long> pids) {
            Map<Long, Map<ModerationDiagType, Set<Long>>> reasonsByPid = new HashMap<>();
            Map<Long, AdGroupCounters> countersByPid = new HashMap<>();

            getAdsStatusesOnAdGroupsForCount(shard, pids).forEach((pid, v) -> {
                var reasons = reasonsByPid.computeIfAbsent(pid, p -> new EnumMap<>(ModerationDiagType.class));
                var counters = countersByPid.computeIfAbsent(pid, p -> new AdGroupCounters());
                for (AggregatedStatusAdData statusData : v) {
                    for (var entry : statusData.getRejectReasons().entrySet()) {
                        reasons.computeIfAbsent(entry.getKey(), k -> new HashSet<>()).addAll(entry.getValue());
                    }
                    counters.incrAdsCount();
                    if (statusData.getStatus().isPresent()) {
                        counters.incrAd(statusData.getStates());
                        counters.incrAd(statusData.getStatus().get());
                    }
                }
            });

            aggregatedStatusesRepository.getKeywordsStatusesOnAdGroupsForCount(shard, pids).forEach((pid, v) -> {
                var counters = countersByPid.computeIfAbsent(pid, p -> new AdGroupCounters());
                for (AggregatedStatusKeywordData statusData : v) {
                    counters.incrKeywordsCount();
                    if (statusData.getStatus().isPresent()) {
                        counters.incrKeyword(statusData.getStates());
                        counters.incrKeyword(statusData.getStatus().get());
                    }
                }
            });

            aggregatedStatusesRepository.getRetargetingsStatusesOnAdGroupsForCount(shard, pids).forEach((pid, v) -> {
                var counters = countersByPid.computeIfAbsent(pid, p -> new AdGroupCounters());
                for (AggregatedStatusRetargetingData statusData : v) {
                    counters.incrRetargetingsCount();
                    if (statusData.getStatus().isPresent()) {
                        counters.incrRetargeting(statusData.getStates());
                        counters.incrRetargeting(statusData.getStatus().get());
                    }
                }
            });

            return new RejectReasonsAndCountersByIds<>(reasonsByPid, countersByPid);
        }

        private Map<Long, List<AggregatedStatusAdData>> getAdsStatusesOnAdGroupsForCount(int shard, @Nullable Set<Long> pids) {
            if (pids == null || pids.isEmpty()) {
                return emptyMap();
            }
            if (!fetchDataFromGrutForAggrStatuses.getOrDefault(false)) {
                return aggregatedStatusesRepository.getAdsStatusesOnAdGroupsForCount(shard, pids);
            }

            Map<Long, Map<Long, AggregatedStatusAdData>> aggregatedStatusAdDataByBidByPid =
                    grutAggregatedStatusService.adsStatusesOnAdGroupsForCount(shard, pids);

            // у данных из mysql приоритет перед грутовыми
            Map<Long, Map<Long, AggregatedStatusAdData>> aggregatedStatusAdDataByBidByPidMysql =
                    aggregatedStatusesRepository.getAdsStatusesOnAdGroupsForCountWithId(shard, pids);
            aggregatedStatusAdDataByBidByPidMysql.forEach((pid, adsStatusesByBid) -> {
                adsStatusesByBid.forEach((bid, adsStatuses) -> {
                    aggregatedStatusAdDataByBidByPid.putIfAbsent(pid, new HashMap<>());
                    aggregatedStatusAdDataByBidByPid.get(pid).put(bid, adsStatuses);
                });
            });

            return EntryStream.of(aggregatedStatusAdDataByBidByPid)
                    .mapValues(adsStatusesByBid -> StreamEx.of(adsStatusesByBid.values()).toList())
                    .toMap();
        }

        private Set<Long> processChangesAds(Collection<ChangesHolder.AdChangesEvent> adEvents,
                                            Set<Long> adsWithAdditionsChanged,
                                            Set<Long> adsWithPerfCreaviesChanged) {
            if (adEvents.isEmpty() && adsWithAdditionsChanged.isEmpty() && adsWithPerfCreaviesChanged.isEmpty()) {
                return Set.of();
            }

            boolean fetchDataFromGrut = fetchDataFromGrutForAggrStatuses.getOrDefault(false);
            Set<Long> allAdIds = new HashSet<>(adsWithAdditionsChanged);
            allAdIds.addAll(adsWithPerfCreaviesChanged);
            allAdIds.addAll(adEvents.stream()
                    .filter(not(ChangesHolder.AdChangesEvent::isDeleted))
                    .map(ChangesHolder.AdChangesEvent::getBannerId)
                    .collect(Collectors.toList()));

            Set<Long> adgroupIdsToRecount = new HashSet<>();

            try (TraceProfile ignore =
                         Trace.current().profile("aggr_statuses:process_ads", "", allAdIds.size())
            ) {
                for (List<Long> bidsChunk : Iterables.partition(allAdIds, BIDS_CHUNK_SIZE)) {
                    var mysqlChunkData =
                            bannerDataForStatusService.bannerDataForStatusByBids(shard, bidsChunk);

                    List<BannerDataForStatus> grutChunkData = emptyList();
                    if (fetchDataFromGrut) {
                        var mysqlBids = StreamEx.of(mysqlChunkData)
                                .map(bannerData -> bannerData.banner().getId())
                                .toSet();
                        var grutBids = StreamEx.of(bidsChunk)
                                .filter(bid -> !mysqlBids.contains(bid))
                                .toList();
                        if (!grutBids.isEmpty()) {
                            logger.info("Got grut bids: {}", grutBids);
                        }
                        grutChunkData = grutAggregatedStatusService.bannerDataForStatusByBidsFromGrut(shard, grutBids);
                        if (grutBids.size() != grutChunkData.size()) {
                            logger.info("Can't find all data in grut, needed: {}, found: {}", grutBids,
                                    StreamEx.of(grutChunkData).map(it -> it.banner().getId()).toList());
                        }
                    }

                    var chunkData = StreamEx.of(mysqlChunkData, grutChunkData)
                            .flatMap(Collection::stream)
                            .toList();

                    final var changes = adChangesProcessor.changes(chunkData);

                    adgroupIdsToRecount.addAll(changes.parentIdsToRecount());
                    aggregatedStatusesRepository.updateAds(shard, updateBefore, changes.newStatuses());
                }
            }
            Set<Long> deletedAdIds = adEvents.stream()
                    .filter(ChangesHolder.AdChangesEvent::isDeleted)
                    .map(ChangesHolder.AdChangesEvent::getBannerId)
                    .collect(Collectors.toSet());

            try (TraceProfile ignore = Trace.current().profile(
                    "aggr_statuses:process_ads_delete", "", deletedAdIds.size())
            ) {
                aggregatedStatusesRepository.deleteAdStatuses(shard, deletedAdIds);
            }
            // В случае, если не нужны id затронутых адгрупп
            if (recalculationDepth == RecalculationDepthEnum.ADGROUPSUBJECTS) {
                return Set.of();
            }

            // Какие-то adgroupId могли быть не добавлены выше, т.к. событие пришло по удаленной записи
            Set<Long> eventFilledPids = adEvents.stream()
                    .map(ChangesHolder.AdChangesEvent::getAdgroupId)
                    .filter(Objects::nonNull).collect(Collectors.toSet());
            // множества пересекаются, но это не страшно, т.к. добавляем в set
            adgroupIdsToRecount.addAll(eventFilledPids);

            return adgroupIdsToRecount;
        }

        private Set<Long> processChangesCallouts(Collection<ChangesHolder.CalloutChangesEvent> calloutEvents) {
            if (calloutEvents.isEmpty()) {
                return Set.of();
            }
            var adIdsToRecompute = new HashSet<Long>();
            for (List<ChangesHolder.CalloutChangesEvent> eventsChunk :
                    Iterables.partition(calloutEvents, CALLOUTS_CHUNK_SIZE)) {
                List<Long> idsChunk = eventsChunk.stream().map(ChangesHolder.CalloutChangesEvent::getAdditionItemId)
                        .collect(Collectors.toList());
                Set<Long> usedInBanners = bannerAdditionsRepository.getBannerIdsByAdditions(shard, idsChunk);
                adIdsToRecompute.addAll(usedInBanners);
            }

            return adIdsToRecompute;
        }

        private Set<Long> processChangesPerfCreatives(Collection<ChangesHolder.PerfCreativeChangesEvent> perfCreativeEvents) {
            if (perfCreativeEvents.isEmpty()) {
                return Set.of();
            }
            var adIdsToRecompute = new HashSet<Long>();
            for (List<ChangesHolder.PerfCreativeChangesEvent> eventsChunk :
                    Iterables.partition(perfCreativeEvents, PERF_CREATIVES_CHUNK_SIZE)) {
                List<Long> idsChunk = eventsChunk.stream().map(ChangesHolder.PerfCreativeChangesEvent::getCreativeId)
                        .collect(Collectors.toList());
                Set<Long> usedInBanners =
                        new HashSet(newBannerCreativeRepository.getBannerPerformanceBannerIds(shard, idsChunk));
                adIdsToRecompute.addAll(usedInBanners);
            }

            return adIdsToRecompute;
        }

        private Set<Long> processRestrictedGeoChanges(Collection<ChangesHolder.RestrictedGeoChangesEvent> restrictedGeoEvents) {
            if (restrictedGeoEvents.isEmpty()) {
                return Set.of();
            }
            List<Long> ids = restrictedGeoEvents.stream().map(ChangesHolder.RestrictedGeoChangesEvent::getBannerId)
                    .collect(Collectors.toList());
            return new HashSet<>(bannerRepository.getAdGroupIdsByBannerIds(shard, ids));
        }

        private Set<Long> processChangesKeywords(Collection<ChangesHolder.KeywordChangesEvent> keywordEvents,
                                                 Collection<ChangesHolder.BidBaseChangesEvent> bidsBaseChanges) {
            if (keywordEvents.isEmpty() && bidsBaseChanges.isEmpty()) {
                return Set.of();
            }

            var keywordIds = keywordEvents.stream()
                    .filter(not(ChangesHolder.KeywordChangesEvent::isDeleted))
                    .map(ChangesHolder.KeywordChangesEvent::getKeywordId)
                    .collect(Collectors.toList());

            Set<Long> adgroupIdsToRecount = new HashSet<>();
            try (TraceProfile ignore = Trace.current().profile(
                    "aggr_statuses:process_keywords", "", keywordIds.size())
            ) {
                for (var keywordIdsChunk : Iterables.partition(keywordIds, KWIDS_CHUNK_SIZE)) {
                    List<StatusAggregationKeyword> keywords = keywordRepository.getKeywords(shard, keywordIdsChunk);

                    Map<Long, AggregatedStatusKeywordData> statuses = new HashMap<>();
                    for (StatusAggregationKeyword keyword : keywords) {
                        Collection<KeywordStatesEnum> states = keywordStates.calc(keyword);
                        SelfStatus selfStatus = calcKeywordSelfStatus(states);
                        statuses.put(keyword.getId(), new AggregatedStatusKeywordData(selfStatus));
                        adgroupIdsToRecount.add(keyword.getAdGroupId());
                    }
                    logger.debug("Updating aggregated statuses for {} keywords, ids: {}", statuses.size(),
                            statuses.keySet());
                    aggregatedStatusesRepository.updateKeywords(shard, updateBefore, statuses);
                }
            }
            var deletedKeywordEvents = keywordEvents.stream()
                    .filter(ChangesHolder.KeywordChangesEvent::isDeleted)
                    .collect(Collectors.toList());

            try (TraceProfile ignore = Trace.current().profile(
                    "aggr_statuses:process_keywords_delete", "", deletedKeywordEvents.size())
            ) {
                for (var deletedKeywordsChunk : Iterables.partition(deletedKeywordEvents, ARC_KWIDS_CHUNK_SIZE)) {
                    // DIRECT-116799
                    var keyTriplets = mapList(deletedKeywordsChunk, keyword ->
                            new AggregatedStatusesArchivedKeywordRepository.CidPidIdTriplet(
                                    keyword.getCampaignId(), keyword.getAdgroupId(), keyword.getKeywordId()));

                    // Не нужно удалять статус у архивированных keyword
                    Set<Long> archivedKeywords = archivedKeywordRepository
                            .getArchivedKeywordsForStatusAggregationByPKeyTriplets(shard, keyTriplets);

                    var actuallyDeletedKeywordIds = deletedKeywordsChunk.stream()
                            .map(ChangesHolder.KeywordChangesEvent::getKeywordId)
                            .filter(id -> !archivedKeywords.contains(id))
                            .collect(Collectors.toSet());

                    aggregatedStatusesRepository.deleteKeywordStatuses(shard, actuallyDeletedKeywordIds);
                }
            }
            adgroupIdsToRecount.addAll(processChangesRelevanceMatchEvents(bidsBaseChanges));

            // В случае, если не нужны id затронутых адгрупп
            if (recalculationDepth == RecalculationDepthEnum.ADGROUPSUBJECTS) {
                return Set.of();
            }

            // Какие-то adgroupId могли быть не добавлены выше, т.к. событие пришло по удаленной записи
            Set<Long> eventFilledPids = keywordEvents.stream()
                    .map(ChangesHolder.KeywordChangesEvent::getAdgroupId)
                    .filter(Objects::nonNull).collect(Collectors.toSet());
            // множества пересекаются, но это не страшно, т.к. добавляем в set
            adgroupIdsToRecount.addAll(eventFilledPids);

            return adgroupIdsToRecount;
        }

        private Set<Long> processChangesRelevanceMatchEvents(
                Collection<ChangesHolder.BidBaseChangesEvent> bidsBaseEvents) {

            if (bidsBaseEvents.isEmpty()) {
                return Set.of();
            }

            for (List<ChangesHolder.BidBaseChangesEvent> bidsBaseEventsChunk : Iterables.partition(bidsBaseEvents,
                    KWIDS_CHUNK_SIZE)) {

                List<Long> bidIdsChunk = bidsBaseEventsChunk.stream()
                        .map(ChangesHolder.BidBaseChangesEvent::getBidId).collect(Collectors.toList());

                List<Bid> relevanceMatchBids = bidRepository.getRelevanceMatchByIdsNotDeleted(shard, bidIdsChunk);

                Map<Long, AggregatedStatusKeywordData> statuses = new HashMap<>();
                for (Bid bid : relevanceMatchBids) {
                    SelfStatus selfStatus = calcRelevanceMatchSelfStatus(bid);
                    statuses.put(bid.getId(), new AggregatedStatusKeywordData(selfStatus));
                }
                logger.debug("Updating aggregated statuses for {} relevanceMatch, ids: {}", statuses.size(),
                        statuses.keySet());
                aggregatedStatusesRepository.updateKeywords(shard, updateBefore, statuses);
            }

            // пересчитываем группы по всем событиям, и по relevance_match и по другим ключевикам, т.к.,
            // в случае полного удаления (через DELETE FROM, не через opts.set(deleted)) записи
            // из bids_base мы уже не можем знать ее тип
            return bidsBaseEvents.stream()
                    .map(ChangesHolder.BidBaseChangesEvent::getAdgroupId)
                    .filter(Objects::nonNull).collect(Collectors.toSet());
        }

        private Set<Long> processChangesRetargetings(
                Collection<ChangesHolder.RetargetingChangesEvent> retargetingEvents) {
            if (retargetingEvents.isEmpty()) {
                return Set.of();
            }

            Set<Long> retIds = retargetingEvents.stream()
                    .filter(not(ChangesHolder.RetargetingChangesEvent::isDeleted))
                    .map(ChangesHolder.RetargetingChangesEvent::getRetargetingId)
                    .collect(Collectors.toSet());

            for (var retIdsChunk : Iterables.partition(retIds, RETIDS_CHUNK_SIZE)) {
                var retargetings = retargetingRepository.getRetargetingsForStatusAggregationByIds(shard, retIdsChunk,
                        maxLimited());

                var statuses = new HashMap<Long, AggregatedStatusRetargetingData>();
                for (var retargeting : retargetings) {
                    Collection<RetargetingStatesEnum> states = retargetingStates.calc(retargeting);
                    var selfStatus = calcRetargetingSelfStatus(states);
                    statuses.put(retargeting.getId(), new AggregatedStatusRetargetingData(states, selfStatus));
                }
                logger.debug("Updating aggregated statuses for {} retargetings, ids: {}", statuses.size(),
                        statuses.keySet());
                aggregatedStatusesRepository.updateRetargetings(shard, updateBefore, statuses);
            }

            Set<Long> deletedRetargetingIds = retargetingEvents.stream()
                    .filter(ChangesHolder.RetargetingChangesEvent::isDeleted)
                    .map(ChangesHolder.RetargetingChangesEvent::getRetargetingId)
                    .collect(Collectors.toSet());

            aggregatedStatusesRepository.deleteRetargetingStatuses(shard, deletedRetargetingIds);

            return recalculationDepth == RecalculationDepthEnum.ADGROUPSUBJECTS
                    ? Set.of()
                    : retargetingEvents.stream()
                    .map(ChangesHolder.RetargetingChangesEvent::getAdGroupId)
                    .filter(Objects::nonNull).collect(Collectors.toSet());
        }

        private Set<Long> processChangesPromoExtensions(
                Collection<ChangesHolder.PromoExtensionChangesEvent> promoExtensionEvents) {
            if (promoExtensionEvents.isEmpty()) {
                return Set.of();
            }
            return cidsAffectedByPromoExtensionChanges(
                    filterAndMapToSet(promoExtensionEvents, e -> true,
                            ChangesHolder.PromoExtensionChangesEvent::getPromoExtensionId));
        }

        private void aggregateOnAdGroupsChange(Collection<Long> pids,
                                               Map<Long, Map<ModerationDiagType, Set<Long>>> updatedRejectReasons,
                                               Map<Long, AdGroupCounters> updatedCounters) {
            Map<Long, AggregatedStatusAdGroupData> adgroupStatuses = new HashMap<>();

            Set<Long> untouchedCountersPids = subtractIds(pids, updatedCounters.keySet());
            var currentRejectReasonsAndCounters =
                    aggregatedStatusesRepository.getAdGroupRejectReasonsAndCountersByIds(shard, untouchedCountersPids);
            var currentRejectReasons = currentRejectReasonsAndCounters.getRejectReasons();
            var currentCounters = currentRejectReasonsAndCounters.getCounters();
            List<AggregatedStatusAdGroup> adGroups = adGroupRepository.getAdGroups(shard, pids);
            adGroupService.enrichAdGroupsWithEffectiveAndRestrictedGeoForStatuses(shard, adGroups);
            Set<Long> foundPids = adGroups.stream()
                    .map(AggregatedStatusAdGroup::getId)
                    .filter(CommonUtils::isValidId)
                    .collect(Collectors.toSet());
            Set<Long> hasInterest = adGroupRepository.getAdGroupIdsWithInterests(shard, foundPids);
            Set<Long> hasGeoLegalFlags = adGroupRepository.getAdGroupIdsWithGeoLegalFlags(shard, foundPids);
            Set<Long> dynamicOrPerformancePids = adGroups.stream()
                    .filter(g -> AdGroupType.DYNAMIC.equals(g.getType()) || AdGroupType.PERFORMANCE.equals(g.getType()))
                    .map(AggregatedStatusAdGroup::getId)
                    .filter(CommonUtils::isValidId)
                    .collect(Collectors.toSet());

            Map<Long, StatusBLGenerated> pidsBlGeneratedStatus = adGroupRepository.getGroupStatusBlGenerated(shard,
                    dynamicOrPerformancePids);
            Map<Long, ShowConditionsCounter> adGroupsShowConditionCounts =
                    adGroupRepository.getAdGroupsShowConditionCounts(shard, dynamicOrPerformancePids);
            for (AggregatedStatusAdGroup adgroup : adGroups) {
                Long pid = adgroup.getId();
                if (pidsBlGeneratedStatus.containsKey(pid)) {
                    adgroup.setStatusBlGenerated(pidsBlGeneratedStatus.get(pid));
                }

                var rejectReasons = updatedRejectReasons.getOrDefault(
                        pid, currentRejectReasons.getOrDefault(pid, emptyMap())
                );
                var counters = updatedCounters.getOrDefault(
                        pid, currentCounters.getOrDefault(pid, new AdGroupCounters())
                );
                var showConditionsCounter = adGroupsShowConditionCounts.getOrDefault(pid, new ShowConditionsCounter());
                adgroupStatuses.put(pid, adgroupStatusData(
                        adgroup, rejectReasons, counters, hasInterest.contains(pid), hasGeoLegalFlags.contains(pid),
                        showConditionsCounter
                ));
            }

            logger.debug("Updating aggregated statuses for {} adgroups, ids: {}", adgroupStatuses.size(),
                    adgroupStatuses.keySet());
            aggregatedStatusesRepository.updateAdGroups(shard, updateBefore, adgroupStatuses);
        }

        private void aggregateOnCampaignsChange(Collection<Long> cids,
                                                Map<Long, Map<ModerationDiagType, Set<Long>>> updatedRejectReasons,
                                                Map<Long, CampaignCounters> updatedCounters) {
            Map<Long, AggregatedStatusCampaignData> campaignStatuses = new HashMap<>();

            Set<Long> untouchedCountersCids = subtractIds(cids, updatedCounters.keySet());
            var currentRejectReasonsAndCounters =
                    aggregatedStatusesRepository.getCampaignRejectReasonsAndCountersByIds(shard, untouchedCountersCids);
            var currentRejectReasons = currentRejectReasonsAndCounters.getRejectReasons();
            var currentCounters = currentRejectReasonsAndCounters.getCounters();
            List<AggregatedStatusCampaign> campaigns = campaignService.getCampaigns(shard, cids);
            Set<Long> walletIds = campaigns.stream().map(AggregatedStatusCampaign::getWalletId)
                    .filter(CommonUtils::isValidId).collect(Collectors.toSet());
            Map<Long, AggregatedStatusWallet> walletsById = walletIds.isEmpty() ? emptyMap()
                    : campaignService.getWallets(shard, walletIds).stream()
                    .collect(Collectors.toMap(AggregatedStatusWallet::getId, Function.identity()));

            var clientIds = listToSet(campaigns, campaign -> ClientId.fromLong(campaign.getClientId()));
            Map<Long, ClientNds> clientNdsById = getClientNdsById(clientIds);
            Map<ClientId, Boolean> pessimizedFlagByClientId = lackOfConversionStatusEnabled.getOrDefault(false) ?
                    pessimizedLoginsService.isClientIdsPessimized(clientIds) :
                    Map.of();
            Map<Long, GoalConversion> pessimizedGoalConversionsCountByCampaignId =
                    aggregatedStatusesHelper.getPessimizedGoalConversionsCount(
                            campaigns, pessimizedFlagByClientId);

            var promoExtensionByCid = promoExtensionRepository.getPromoExtensionsByCids(
                    shard, listToSet(campaigns, AggregatedStatusCampaign::getId));

            for (AggregatedStatusCampaign campaign : campaigns) { // если кампании уже нет в БД, то не считаем ее
                Long cid = campaign.getId();
                var rejectReasons = updatedRejectReasons.getOrDefault(
                        cid, currentRejectReasons.getOrDefault(cid, emptyMap())
                );
                var counters = updatedCounters.getOrDefault(
                        cid, currentCounters.getOrDefault(cid, new CampaignCounters())
                );

                AggregatedStatusWallet wallet = walletsById.get(campaign.getWalletId());
                if (isValidId(campaign.getWalletId()) && !walletsById.containsKey(campaign.getWalletId())) {
                    logger.error("No wallet id: {} found for campaign cid: {}, dropping campaign",
                            campaign.getWalletId(), campaign.getId());
                    continue;
                }

                campaignStatuses.put(cid, campaignStatusData(campaign, wallet, rejectReasons, counters,
                        clientNdsById.get(campaign.getClientId()),
                        pessimizedFlagByClientId.getOrDefault(ClientId.fromLong(campaign.getClientId()), false),
                        pessimizedGoalConversionsCountByCampaignId.get(campaign.getId()),
                        promoExtensionByCid.get(campaign.getId())));
            }
            logger.debug("Updating aggregated statuses for {} campaigns, ids: {}", campaignStatuses.size(),
                    campaignStatuses.keySet());
            aggregatedStatusesRepository.updateCampaigns(shard, updateBefore, campaignStatuses);
        }

        private Map<Long, ClientNds> getClientNdsById(Set<ClientId> clientIds) {
            List<Client> clients = clientService.getClients(shard, clientIds);
            List<ClientNds> clientNdsList = clientNdsService.massGetEffectiveClientNds(clients);
            return listToMap(clientNdsList, ClientNds::getClientId);
        }

        private AggregatedStatusCampaignData campaignStatusData(AggregatedStatusCampaign campaign,
                                                                @Nullable AggregatedStatusWallet wallet,
                                                                Map<ModerationDiagType, Set<Long>> rejectReasons,
                                                                CampaignCounters counters,
                                                                @Nullable ClientNds clientNds,
                                                                boolean isClientPessimized,
                                                                @Nullable GoalConversion goalConversion,
                                                                @Nullable PromoExtension promoExtension) {
            boolean hasWallet = isValidId(campaign.getWalletId());
            Collection<CampaignStatesEnum> states = campaignStates.calc(campaign, wallet,
                    ifNotNull(clientNds, ClientNds::getNds),
                    isClientPessimized,
                    ifNotNull(goalConversion, GoalConversion::getGoals),
                    PromoactionsStatusmoderate.No.equals(ifNotNull(promoExtension, PromoExtension::getStatusModerate)));
            SelfStatus selfStatus = calcCampaignSelfStatus(campaign.getId(), hasWallet, states, rejectReasons,
                    counters, campaign.getIsDraftApproveAllowed(), useNewSelfStatusCalculator);
            return new AggregatedStatusCampaignData(states, counters, selfStatus);
        }

        private AggregatedStatusAdGroupData adgroupStatusData(AggregatedStatusAdGroup adgroup,
                                                              Map<ModerationDiagType, Set<Long>> rejectReasons,
                                                              AdGroupCounters counters,
                                                              boolean hasInterest,
                                                              boolean hasBannerGeoLegalFlag,
                                                              ShowConditionsCounter showConditionsCounter) {
            Collection<AdGroupStatesEnum> states = adGroupStates.calc(adgroup);
            SelfStatus selfStatus = calcAdGroupSelfStatus(adgroup, states, rejectReasons, counters, hasInterest,
                    hasBannerGeoLegalFlag, showConditionsCounter, useNewSelfStatusCalculator);
            states.addAll(adGroupStates.calcByStatusAndCounters(selfStatus, counters));
            return new AggregatedStatusAdGroupData(states, counters, selfStatus);
        }

    }

}
