package ru.yandex.direct.core.aggregatedstatuses;

import java.time.Instant;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.logic.RejectedReasons.EntitiesRejectedReasons;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesAdGroupRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.AdWithStatus;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.KeywordWithStatus;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RetargetingWithStatus;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.aggregatedstatuses.AggregatedStatusBaseData;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusWallet;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatusInfo;
import ru.yandex.direct.core.entity.campaign.service.TimeTargetStatusService;
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.utils.CommonUtils;

import static org.springframework.util.CollectionUtils.containsAny;
import static ru.yandex.direct.core.aggregatedstatuses.logic.EffectiveStatusCalculators.adEffectiveStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.EffectiveStatusCalculators.adGroupEffectiveStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.EffectiveStatusCalculators.campaignEffectiveStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.EffectiveStatusCalculators.keywordEffectiveStatus;
import static ru.yandex.direct.core.aggregatedstatuses.logic.EffectiveStatusCalculators.retargetingEffectiveStatus;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_BL_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для отображения результатов агрегации по статусам на фронт
 * Подробнее о системе статусов см. документацию: https://docs.yandex-team.ru/direct-dev/aggr-statuses/concept.html
 */
@Service
@ParametersAreNonnullByDefault
public class AggregatedStatusesViewService implements AggregatedStatuses {
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final AggregatedStatusesCampaignService aggregatedStatusesCampaignService;
    private final AggregatedStatusesAdGroupRepository aggregatedStatusesAdGroupRepository;
    private final TimeTargetStatusService timeTargetStatusService;
    private final GeoTimezoneMappingService geoTimezoneMappingService;
    private final ObjectMapper objectMapper;
    private final ShardHelper shardHelper;

    @Autowired
    public AggregatedStatusesViewService(AggregatedStatusesRepository aggregatedStatusesRepository,
                                         AggregatedStatusesCampaignService aggregatedStatusesCampaignService,
                                         AggregatedStatusesAdGroupRepository aggregatedStatusesAdGroupRepository,
                                         TimeTargetStatusService timeTargetStatusService,
                                         GeoTimezoneMappingService geoTimezoneMappingService,
                                         ShardHelper shardHelper) {
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.aggregatedStatusesCampaignService = aggregatedStatusesCampaignService;
        this.aggregatedStatusesAdGroupRepository = aggregatedStatusesAdGroupRepository;
        this.timeTargetStatusService = timeTargetStatusService;
        this.geoTimezoneMappingService = geoTimezoneMappingService;
        this.shardHelper = shardHelper;
        this.objectMapper = new ObjectMapper();
        this.objectMapper.registerModule(new Jdk8Module());
    }

    public Map<Long, AggregatedStatusAdData> getAdStatusesByIds(int shard, Set<Long> bids) {
        var statusesData = aggregatedStatusesRepository.getAdStatusesDataForView(shard, bids);
        return toAdStatusDataMap(shard, statusesData);
    }

    public Map<Long, AggregatedStatusAdData> getAdStatusesByAdgroupIds(int shard, ClientId clientId, Collection<Long> adgroupIds) {
        var statusesData = aggregatedStatusesRepository.getAdStatusesDataByPidsForPopup(shard, clientId, adgroupIds);
        return toAdStatusDataMap(shard, statusesData);
    }

    private Map<Long, AggregatedStatusAdData> toAdStatusDataMap(int shard, List<AdWithStatus> statusesData) {
        var pids = statusesData.stream().map(AdWithStatus::getAdGroupId).collect(Collectors.toSet());
        var adGroupStatusesByIds = getAdGroupStatusesByIds(shard, pids);

        Map<Long, AggregatedStatusAdData> result = new HashMap<>();
        for (var st : statusesData) {
            var adGroupStatusData = adGroupStatusesByIds.get(st.getAdGroupId());
            var statusData = st.getStatusData();
            SelfStatus adStatus = adEffectiveStatus(statusData, adGroupStatusData);
            if (adStatus != null) {
                statusData.updateSelfStatus(adStatus);
                result.put(st.getAdId(), statusData);
            }
        }
        return result;
    }

    public Map<Long, AggregatedStatusAdGroupData> getAdGroupStatusesByCampaignId(int shard, ClientId clientId,
                                                                                 Long campaignId) {
        Instant currentInstant = Instant.now();
        List<AggregatedStatusAdGroup> adGroupsWithStatuses =
                aggregatedStatusesAdGroupRepository.getAdGroupsWithStatusesByCidForPopup(shard, clientId, campaignId);
        Map<Long, AggregatedStatusAdGroupData> result = toAdGroupStatusDataMap(shard, currentInstant,
                adGroupsWithStatuses);
        return result;
    }

    public Map<Long, AggregatedStatusAdGroupData> getAdGroupStatusesByIds(int shard, Set<Long> pids) {
        Instant currentInstant = Instant.now();
        List<AggregatedStatusAdGroup> adGroupsWithStatuses =
                aggregatedStatusesAdGroupRepository.getAdGroupsWithStatusesForView(shard, pids);
        Map<Long, AggregatedStatusAdGroupData> result = toAdGroupStatusDataMap(shard, currentInstant,
                adGroupsWithStatuses);
        return result;
    }

    private Map<Long, AggregatedStatusAdGroupData> toAdGroupStatusDataMap(int shard, Instant currentInstant,
                                                                          List<AggregatedStatusAdGroup> adGroupsWithStatuses) {
        Set<AdGroupType> dynamicAndSmartAdGroupTypes = Set.of(AdGroupType.DYNAMIC, AdGroupType.PERFORMANCE);
        Set<GdSelfStatusReason> blProcessingReasons =
                Set.of(ADGROUP_BL_PROCESSING, ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN);
        Set<Long> cids = adGroupsWithStatuses.stream().map(st -> st.getCampaignId()).collect(Collectors.toSet());
        Map<Long, AggregatedStatusCampaignData> campaignStatusesByIds = getCampaignStatusesByIds(shard, cids);
        Map<Long, AggregatedStatusAdGroupData> result = new HashMap<>();
        for (var g : adGroupsWithStatuses) {
            var campaignStatusData = campaignStatusesByIds.get(g.getCampaignId());
            AggregatedStatusAdGroupData statusData = g.getAggregatedStatus();
            SelfStatus adGroupStatus = adGroupEffectiveStatus(currentInstant, g, campaignStatusData);
            if (adGroupStatus != null) {
                statusData.updateSelfStatus(adGroupStatus);
                if (GdSelfStatusEnum.allOk().contains(adGroupStatus.getStatus())
                        || (
                        statusData.getCounters() != null // just in case, but shouldn't be null here
                                && statusData.getCounters().getKeywordsTotal() == 0
                                && statusData.getCounters().getAdsTotal() == 0
                                && statusData.getCounters().getRetargetingsTotal() == 0)
                        || (
                        dynamicAndSmartAdGroupTypes.contains(g.getType())
                                && containsAny(statusData.getReasons(), blProcessingReasons))) {
                    statusData.flushCounters();
                }
                result.put(g.getId(), statusData);
            }
        }
        return result;
    }

    public Map<Long, AggregatedStatusCampaignData> getCampaignStatusesByIds(ClientId clientId, Collection<Long> cids) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getCampaignStatusesByIds(shard, cids);
    }

    public Map<Long, AggregatedStatusCampaignData> getCampaignStatusesByIds(int shard, Collection<Long> cids) {
        var currentInstant = Instant.now();
        var subCidsById = aggregatedStatusesCampaignService.getSubCampaignIdsByMasterId(shard, cids);
        var allCids = StreamEx.<Collection<Long>>of(subCidsById.values())
                .append(cids)
                .toFlatCollection(Function.identity(), HashSet::new);
        var campaignById = aggregatedStatusesCampaignService.getCampaignById(shard, allCids);
        var walletIds = mapAndFilterToSet(
                campaignById.values(), AggregatedStatusCampaign::getWalletId, CommonUtils::isValidId);
        var walletById = listToMap(
                aggregatedStatusesCampaignService.getWallets(shard, walletIds), AggregatedStatusWallet::getId);
        var campaigns = mapAndFilterList(cids, campaignById::get, Objects::nonNull);

        Map<Long, AggregatedStatusCampaignData> result = new HashMap<>();
        for (var campaign : campaigns) {
            AggregatedStatusCampaignData aggregatedStatus = campaign.getAggregatedStatus();
            if (aggregatedStatus == null || aggregatedStatus.getStatus().isEmpty()) {
                continue;
            }
            List<AggregatedStatusCampaign> subCampaigns = mapList(
                    subCidsById.getOrDefault(campaign.getId(), List.of()), campaignById::get);
            TimeTargetStatusInfo timeTargetStatus =
                    timeTargetStatusService.getTimeTargetStatus(campaign.getTimeTarget(),
                            geoTimezoneMappingService.getRegionIdByTimezoneId(campaign.getTimezoneId()),
                            currentInstant);
            SelfStatus status = campaignEffectiveStatus(currentInstant, campaign, subCampaigns,
                    walletById.get(campaign.getWalletId()), timeTargetStatus);
            aggregatedStatus.updateSelfStatus(status);
            if (GdSelfStatusEnum.allOk().contains(status.getStatus())
                    || aggregatedStatus.getCounters().getAdgroupsTotal() == 0) {
                aggregatedStatus.flushCounters();
            }
            result.put(campaign.getId(), aggregatedStatus);
        }
        return result;
    }

    public Map<Long, AggregatedStatusKeywordData> getKeywordStatusesByIds(int shard, Set<Long> adgroupIds,
                                                                          Set<Long> keywordIds) {
        var statusesData = aggregatedStatusesRepository.getKeywordStatusesDataForView(shard, adgroupIds, keywordIds);
        return toKeywordStatusesDataMap(shard, statusesData);
    }

    public Map<Long, AggregatedStatusKeywordData> getKeywordStatusesByAdgroupIds(int shard, ClientId clientId,
                                                                                Collection<Long> adgroupIds) {
        var statusesData = aggregatedStatusesRepository.getKeywordStatusesDataByPidsForPopup(shard, clientId, adgroupIds);
        return toKeywordStatusesDataMap(shard, statusesData);
    }

    private Map<Long, AggregatedStatusKeywordData> toKeywordStatusesDataMap(int shard,
                                                                            List<KeywordWithStatus> statusesData) {
        var pids = statusesData.stream().map(KeywordWithStatus::getAdGroupId).collect(Collectors.toSet());
        var adGroupStatusesByIds = getAdGroupStatusesByIds(shard, pids);

        Map<Long, AggregatedStatusKeywordData> result = new HashMap<>();
        for (var st : statusesData) {
            var adGroupStatusData = adGroupStatusesByIds.get(st.getAdGroupId());
            var statusData = st.getStatusData();
            SelfStatus keywordStatus = keywordEffectiveStatus(statusData, adGroupStatusData);
            if (keywordStatus != null) {
                statusData.updateSelfStatus(keywordStatus);
                result.put(st.getKeywordId(), statusData);
            }
        }
        return result;
    }

    public Map<Long, AggregatedStatusRetargetingData> getRetargetingStatusesByIds(int shard, Set<Long> retargetingIds) {
        var statusesData = aggregatedStatusesRepository.getRetargetingStatusesDataForView(shard, retargetingIds);
        return getRetargetingStatusesDataMap(shard, statusesData);
    }

    public Map<Long, AggregatedStatusRetargetingData> getRetargetingStatusesByAdgroupIds(int shard, ClientId clientId,
                                                                                        Collection<Long> adgroupIds) {
        var statusesData = aggregatedStatusesRepository.getRetargetingStatusesDataByPidsForPopup(shard, clientId,
                adgroupIds);
        return getRetargetingStatusesDataMap(shard, statusesData);
    }

    private Map<Long, AggregatedStatusRetargetingData> getRetargetingStatusesDataMap(int shard,
                                                                                     List<RetargetingWithStatus> statusesData) {
        var pids = statusesData.stream().map(RetargetingWithStatus::getAdGroupId).collect(Collectors.toSet());
        var adGroupStatusesByIds = getAdGroupStatusesByIds(shard, pids);

        var result = new HashMap<Long, AggregatedStatusRetargetingData>();
        for (var st : statusesData) {
            var adGroupStatusData = adGroupStatusesByIds.get(st.getAdGroupId());
            var statusData = st.getStatusData();
            var retargetingStatus = retargetingEffectiveStatus(statusData, adGroupStatusData);
            if (retargetingStatus != null) {
                statusData.updateSelfStatus(retargetingStatus);
                result.put(st.getRetargetingId(), statusData);
            }
        }
        return result;
    }

    public Map<Long, String> statusesToJsonString(Map<Long, ? extends AggregatedStatusBaseData> statuses) {
        return EntryStream.of(statuses).mapValues(v -> toJson(v)).toMap();
    }

    public <T extends AggregatedStatusBaseData> String toJson(T statusData) {
        try {
            return objectMapper.writeValueAsString(statusData);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    // На объявлении могут быть причины отклонения. Правильнее считать по States, но по reasons надежнее в том
    // смысле, что не получится ситуации когда мы выведем подсказку с причинами отклонения, а более никак
    // не покажем что на объявлении что-то не так с модерацией.
    public boolean adCouldHaveRejectReasons(AggregatedStatusAdData statusData) {
        return EntitiesRejectedReasons.adRejectedReasons.hasRejectedReason(statusData.getReasons());
    }

    public boolean adgroupCouldHaveRejectReasons(AggregatedStatusAdGroupData statusData) {
        return EntitiesRejectedReasons.adGroupRejectedReasons.hasRejectedReason(statusData.getReasons())
                || statusData.getStates().contains(AdGroupStatesEnum.REJECTED)
                || (statusData.getCounters() != null && statusData.getCounters().getKeywordStatuses()
                .containsKey(GdSelfStatusEnum.STOP_CRIT));
    }

    public boolean campaignCouldHaveRejectReasons(AggregatedStatusCampaignData statusData) {
        return EntitiesRejectedReasons.campaignRejectedReasons.hasRejectedReason(statusData.getReasons())
                || statusData.getStates().contains(CampaignStatesEnum.REJECTED);
    }
}
