package ru.yandex.direct.core.aggregatedstatuses;

import java.util.ArrayList;
import java.util.Collection;
import java.util.function.Consumer;

import one.util.streamex.StreamEx;

import static com.google.common.collect.Iterables.partition;
import static java.util.Collections.emptyList;

public class ChangesHolder {
    private final Collection<CampaignChangesEvent> campaignsChanges;
    private final Collection<AdgroupChangesEvent> adgroupsChanges;
    private final Collection<AdChangesEvent> adsChanges;
    private final Collection<KeywordChangesEvent> keywordsChanges;
    private final Collection<BidBaseChangesEvent> bidsBaseChanges;
    private final Collection<CalloutChangesEvent> calloutsChanges;
    private final Collection<PerfCreativeChangesEvent> perfCreativesChanges;
    private final Collection<RestrictedGeoChangesEvent> restrictedGeoChanges;
    private final Collection<RetargetingChangesEvent> retargetingsChanges;
    private final Collection<PromoExtensionChangesEvent> promoExtensionChanges;

    public ChangesHolder(Collection<CampaignChangesEvent> campaignsChanges,
                         Collection<AdgroupChangesEvent> adgroupsChanges, Collection<AdChangesEvent> adsChanges,
                         Collection<KeywordChangesEvent> keywordsChanges,
                         Collection<BidBaseChangesEvent> bidsBaseChanges,
                         Collection<CalloutChangesEvent> calloutsChanges,
                         Collection<PerfCreativeChangesEvent> perfCreativesChanges,
                         Collection<RestrictedGeoChangesEvent> restrictedGeoChanges,
                         Collection<RetargetingChangesEvent> retargetingsChanges,
                         Collection<PromoExtensionChangesEvent> promoExtensionChanges) {
        this.campaignsChanges = campaignsChanges;
        this.adgroupsChanges = adgroupsChanges;
        this.adsChanges = adsChanges;
        this.keywordsChanges = keywordsChanges;
        this.bidsBaseChanges = bidsBaseChanges;
        this.calloutsChanges = calloutsChanges;
        this.perfCreativesChanges = perfCreativesChanges;
        this.restrictedGeoChanges = restrictedGeoChanges;
        this.retargetingsChanges = retargetingsChanges;
        this.promoExtensionChanges = promoExtensionChanges;
    }

    public static CampaignChangesEvent campaignChangesEvent(Long cid, boolean deleted) {
        return new CampaignChangesEvent(cid, deleted);
    }

    public static AdChangesEvent adChangesEvent(Long bid, Long cid, boolean deleted) {
        return new AdChangesEvent(bid, cid, deleted);
    }

    public static AdgroupChangesEvent adgroupChangesEvent(Long pid, Long cid, boolean deleted) {
        return new AdgroupChangesEvent(pid, cid, deleted);
    }

    public static KeywordChangesEvent keywordChangesEvent(Long id, Long pid, Long cid, boolean deleted) {
        return new KeywordChangesEvent(id, pid, cid, deleted);
    }

    // У объектов relevance match из bids_base есть статус, но удаление оттуда невозможно отличить
    // от архивации keyword (нету поля type и/или части ключа чтобы посмотреть в bids_arc),
    // поэтому статусы relevance match не удаляем
    public static BidBaseChangesEvent bidBaseChangesEvent(Long id, Long pid) {
        return new BidBaseChangesEvent(id, pid);
    }

    public static CalloutChangesEvent calloutChangesEvent(Long additionItemId) {
        return new CalloutChangesEvent(additionItemId);
    }

    public static PerfCreativeChangesEvent perfCreativeChangesEvent(Long creativeId) {
        return new PerfCreativeChangesEvent(creativeId);
    }

    public static RestrictedGeoChangesEvent restrictedGeoChangesEvent(Long bid) {
        return new RestrictedGeoChangesEvent(bid);
    }

    public static RetargetingChangesEvent retargetingChangesEvent(Long retargetingId, Long pid, boolean deleted) {
        return new RetargetingChangesEvent(retargetingId, pid, deleted);
    }

    public static PromoExtensionChangesEvent promoExtensionChangesEvent(Long promoExtensionId) {
        return new PromoExtensionChangesEvent(promoExtensionId);
    }

    public Collection<CampaignChangesEvent> getCampaignsChanges() {
        return campaignsChanges;
    }

    public Collection<AdgroupChangesEvent> getAdgroupsChanges() {
        return adgroupsChanges;
    }

    public Collection<AdChangesEvent> getAdsChanges() {
        return adsChanges;
    }

    public Collection<KeywordChangesEvent> getKeywordsChanges() {
        return keywordsChanges;
    }

    public Collection<BidBaseChangesEvent> getBidsBaseChanges() {
        return bidsBaseChanges;
    }

    public Collection<CalloutChangesEvent> getCalloutsChanges() {
        return calloutsChanges;
    }

    public Collection<PerfCreativeChangesEvent> getPerfCreativesChanges() {
        return perfCreativesChanges;
    }

    public Collection<RestrictedGeoChangesEvent> getRestrictedGeoChanges() {
        return restrictedGeoChanges;
    }

    public Collection<RetargetingChangesEvent> getRetargetingsChanges() {
        return retargetingsChanges;
    }

    public Collection<PromoExtensionChangesEvent> getPromoExtensionChanges() {
        return promoExtensionChanges;
    }

    @Override
    public String toString() {
        var sb = new StringBuilder("ChangesHolder: ");
        if (!campaignsChanges.isEmpty()) {
            sb.append(StreamEx.of(campaignsChanges).map(c -> String.valueOf(c.getCampaignId()))
                    .joining(",", "cids: [", "]\n"));
        }
        if (!adgroupsChanges.isEmpty()) {
            sb.append(StreamEx.of(adgroupsChanges).map(g -> String.valueOf(g.getAdgroupId()))
                    .joining(",", "pids: [", "]\n"));
        }
        if (!adsChanges.isEmpty()) {
            sb.append(StreamEx.of(adsChanges).map(ad -> String.valueOf(ad.getBannerId()))
                    .joining(",", "bids: [", "]\n"));
        }
        if (!keywordsChanges.isEmpty()) {
            sb.append(StreamEx.of(keywordsChanges).map(kw -> String.valueOf(kw.getKeywordId()))
                    .joining(",", "ids: [", "]\n"));
        }
        if (!bidsBaseChanges.isEmpty()) {
            sb.append(StreamEx.of(bidsBaseChanges).map(bid -> String.valueOf(bid.getBidId()))
                    .joining(",", "bids_base.ids: [", "]\n"));
        }
        if (!calloutsChanges.isEmpty()) {
            sb.append(StreamEx.of(calloutsChanges).map(a -> String.valueOf(a.getAdditionItemId()))
                    .joining(",", "additions.ids: [", "]\n"));
        }
        if (!restrictedGeoChanges.isEmpty()) {
            sb.append(StreamEx.of(restrictedGeoChanges).map(b -> String.valueOf(b.getBannerId()))
                    .joining(",", "restricted_geo.bids: [", "]\n"));
        }
        if (!retargetingsChanges.isEmpty()) {
            sb.append(StreamEx.of(retargetingsChanges).map(r -> String.valueOf(r.getRetargetingId()))
                    .joining(",", "bids_retargeting.ret_ids: [", "]\n"));
        }
        if (!promoExtensionChanges.isEmpty()) {
            sb.append(StreamEx.of(promoExtensionChanges).map(p -> String.valueOf(p.getPromoExtensionId()))
                    .joining(",", "promo_extension.ids: [", "]\n"));
        }

        return sb.toString();
    }

    public static class CampaignChangesEvent {
        private Long campaignId;
        private boolean deleted;

        public CampaignChangesEvent(Long campaignId, boolean deleted) {
            this.campaignId = campaignId;
            this.deleted = deleted;
        }

        public Long getCampaignId() {
            return campaignId;
        }

        public boolean isDeleted() {
            return deleted;
        }
    }

    public static class AdChangesEvent {
        private Long bannerId;
        private Long adgroupId;
        private boolean deleted;

        public AdChangesEvent(Long bannerId, Long adgroupId, boolean deleted) {
            this.bannerId = bannerId;
            this.adgroupId = adgroupId;
            this.deleted = deleted;
        }

        public Long getBannerId() {
            return bannerId;
        }

        public Long getAdgroupId() {
            return adgroupId;
        }

        public boolean isDeleted() {
            return deleted;
        }
    }

    public static class AdgroupChangesEvent {
        private Long adgroupId;
        private Long campaignId;
        private boolean deleted;

        public AdgroupChangesEvent(Long adgroupId, Long campaignId, boolean deleted) {
            this.adgroupId = adgroupId;
            this.campaignId = campaignId;
            this.deleted = deleted;
        }

        public Long getAdgroupId() {
            return adgroupId;
        }

        public Long getCampaignId() {
            return campaignId;
        }

        public boolean isDeleted() {
            return deleted;
        }
    }

    public static class KeywordChangesEvent {
        private Long keywordId;
        private Long adgroupId;
        private Long campaignId;
        private boolean deleted;

        public KeywordChangesEvent(Long keywordId, Long adgroupId, Long campaignId, boolean deleted) {
            this.keywordId = keywordId;
            this.adgroupId = adgroupId;
            this.campaignId = campaignId;
            this.deleted = deleted;
        }

        public Long getKeywordId() {
            return keywordId;
        }

        public Long getAdgroupId() {
            return adgroupId;
        }

        public Long getCampaignId() {
            return campaignId;
        }

        public boolean isDeleted() {
            return deleted;
        }
    }

    public static class BidBaseChangesEvent {
        private Long bidId;
        private Long adgroupId;

        public BidBaseChangesEvent(Long bidId, Long adgroupId) {
            this.bidId = bidId;
            this.adgroupId = adgroupId;
        }

        public Long getBidId() {
            return bidId;
        }

        public Long getAdgroupId() {
            return adgroupId;
        }
    }

    public static class CalloutChangesEvent {
        private Long additionItemId;

        public CalloutChangesEvent(Long additionItemId) {
            this.additionItemId = additionItemId;
        }

        public Long getAdditionItemId() {
            return additionItemId;
        }
    }

    public static class PerfCreativeChangesEvent {
        private Long creativeId;

        public PerfCreativeChangesEvent(Long creativeId) { this.creativeId = creativeId; }

        public Long getCreativeId() { return creativeId; }
     }

    public static class RestrictedGeoChangesEvent {
        private Long bannerId;

        public RestrictedGeoChangesEvent(Long bannerId) {
            this.bannerId = bannerId;
        }

        public Long getBannerId() {
            return bannerId;
        }
    }

    public static class RetargetingChangesEvent {
        private Long retargetingId;
        private Long adGroupId;
        private boolean deleted;

        public RetargetingChangesEvent(Long retargetingId, Long adGroupId, boolean deleted) {
            this.retargetingId = retargetingId;
            this.adGroupId = adGroupId;
            this.deleted = deleted;
        }

        public Long getRetargetingId() {
            return retargetingId;
        }

        public Long getAdGroupId() {
            return adGroupId;
        }

        public boolean isDeleted() {
            return deleted;
        }
    }

    public static class PromoExtensionChangesEvent {
        private Long promoExtensionId;

        public PromoExtensionChangesEvent(Long promoExtensionId) {
            this.promoExtensionId = promoExtensionId;
        }

        public Long getPromoExtensionId() {
            return promoExtensionId;
        }
    }

    public static class ChangesHolderLogger {
        private static final int IDS_CHUNK_SIZE = 10000;

        private final Consumer<String> logger;

        private Collection<String> campaignsIdStrings;
        private Collection<String> adgroupsIdStrings;
        private Collection<String> adsIdStrings;
        private Collection<String> keywordsIdStrings;
        private Collection<String> archivedKeywordsIdStrings;
        private Collection<String> bidsBaseIdStrings;
        private Collection<String> calloutsIdStrings;
        private Collection<String> restrictedGeoBannerIdStrings;
        private Collection<String> retargetingsIdStrings;

        public ChangesHolderLogger(ChangesHolder changesHolder, Consumer<String> logger) {
            this.logger = logger;

            campaignsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getCampaignsChanges(), IDS_CHUNK_SIZE)) {
                campaignsIdStrings.add(StreamEx.of(chunk).map(c -> String.valueOf(c.getCampaignId()))
                        .joining(",", String.format("%d cids: [", chunk.size()), "]"));
            }

            adgroupsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getAdgroupsChanges(), IDS_CHUNK_SIZE)) {
                adgroupsIdStrings.add(StreamEx.of(chunk).map(g -> String.valueOf(g.getAdgroupId()))
                        .joining(",", String.format("%d pids: [", chunk.size()), "]"));
            }

            adsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getAdsChanges(), IDS_CHUNK_SIZE)) {
                adsIdStrings.add(StreamEx.of(chunk).map(ad -> String.valueOf(ad.getBannerId()))
                        .joining(",", String.format("%d bids: [", chunk.size()), "]"));
            }

            keywordsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getKeywordsChanges(), IDS_CHUNK_SIZE)) {
                keywordsIdStrings.add(StreamEx.of(chunk).map(kw -> String.valueOf(kw.getKeywordId()))
                        .joining(",", String.format("%d ids: [", chunk.size()), "]"));
            }

            bidsBaseIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getBidsBaseChanges(), IDS_CHUNK_SIZE)) {
                bidsBaseIdStrings.add(StreamEx.of(chunk).map(kw -> String.valueOf(kw.getBidId()))
                        .joining(",", String.format("%d bids_base.ids: [", chunk.size()), "]"));
            }

            calloutsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getCalloutsChanges(), IDS_CHUNK_SIZE)) {
                calloutsIdStrings.add(StreamEx.of(chunk).map(c -> String.valueOf(c.getAdditionItemId()))
                        .joining(",", String.format("%d additions.ids: [", chunk.size()), "]"));
            }

            restrictedGeoBannerIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getRestrictedGeoChanges(), IDS_CHUNK_SIZE)) {
                restrictedGeoBannerIdStrings.add(StreamEx.of(chunk).map(c -> String.valueOf(c.getBannerId()))
                        .joining(",", String.format("%d restricted_geo.bids: [", chunk.size()), "]"));
            }

            retargetingsIdStrings = new ArrayList<>();
            for (var chunk : partition(changesHolder.getRetargetingsChanges(), IDS_CHUNK_SIZE)) {
                retargetingsIdStrings.add(StreamEx.of(chunk).map(r -> String.valueOf(r.getRetargetingId()))
                        .joining(",", String.format("%d bids_retargeting.ret_ids: [", chunk.size()), "]"));
            }
        }

        public void logCampaignsIdStrings() {
            for (var idsString : campaignsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logAdGroupsIdStrings() {
            for (var idsString : adgroupsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logAdsIdStrings() {
            for (var idsString : adsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logKeywordsIdStrings() {
            for (var idsString : keywordsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logBidsBaseIdStrings() {
            for (var idsString : bidsBaseIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logCalloutsIdStrings() {
            for (var idsString : calloutsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logRestrictedGeoBannersIdStrings() {
            for (var idsString : restrictedGeoBannerIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logRetargetingsIdStrings() {
            for (var idsString : retargetingsIdStrings) {
                logger.accept(idsString);
            }
        }

        public void logAllIdStrings() {
            logCampaignsIdStrings();
            logAdGroupsIdStrings();
            logAdsIdStrings();
            logKeywordsIdStrings();
            logBidsBaseIdStrings();
            logCalloutsIdStrings();
            logRestrictedGeoBannersIdStrings();
            logRetargetingsIdStrings();
        }
    }

    public static class Builder {
        private Collection<CampaignChangesEvent> campaignsChanges;
        private Collection<AdgroupChangesEvent> adgroupsChanges;
        private Collection<AdChangesEvent> adsChanges;
        private Collection<KeywordChangesEvent> keywordsChanges;
        private Collection<BidBaseChangesEvent> bidsBaseChanges;
        private Collection<CalloutChangesEvent> calloutsChanges;
        private Collection<PerfCreativeChangesEvent> perfCreativesChanges;
        private Collection<RestrictedGeoChangesEvent> restrictedGeoChanges;
        private Collection<RetargetingChangesEvent> retargetingsChanges;
        private Collection<PromoExtensionChangesEvent> promoExtensionChanges;

        public Builder() {
            campaignsChanges = emptyList();
            adgroupsChanges = emptyList();
            adsChanges = emptyList();
            keywordsChanges = emptyList();
            bidsBaseChanges = emptyList();
            calloutsChanges = emptyList();
            perfCreativesChanges = emptyList();
            restrictedGeoChanges = emptyList();
            retargetingsChanges = emptyList();
            promoExtensionChanges = emptyList();
        }

        public Builder withAdsChanges(Collection<AdChangesEvent> adsChanges) {
            this.adsChanges = adsChanges;
            return this;
        }

        public ChangesHolder build() {
            return new ChangesHolder(campaignsChanges, adgroupsChanges, adsChanges, keywordsChanges, bidsBaseChanges,
                    calloutsChanges, perfCreativesChanges, restrictedGeoChanges, retargetingsChanges, promoExtensionChanges);
        }
    }
}
