package ru.yandex.direct.core.aggregatedstatuses.grut

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.core.aggregatedstatuses.BannerDataForStatus
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.BannerGrutApi
import ru.yandex.direct.core.grut.converter.convertToBannerDataForStatus
import ru.yandex.grut.objects.proto.client.Schema.EObjectType

@Service
class GrutAggregatedStatusService @Autowired constructor(
    grutContext: GrutContext,
    private val aggregatedStatusesRepository: AggregatedStatusesRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
) {

    private val bannerApi = BannerGrutApi(grutContext)
    private val bannerV2Api = BannerGrutApi(grutContext = grutContext, objectType = EObjectType.OT_BANNER_V2)

    fun adsStatusesOnAdGroupsForCount(
        shard: Int,
        adGroupIds: Set<Long>,
    ): Map<Long, Map<Long, AggregatedStatusAdData>> {
        val bannerIdToAdGroupId = bannerApi.getBannersByAdGroupIds(adGroupIds.toList())
            .associate { it.meta.directId to it.meta.adGroupId }

        return aggregatedStatusesRepository.getAdStatusesByIds(shard, bannerIdToAdGroupId.keys).entries
            .groupBy { bannerIdToAdGroupId[it.key]!! }
            .mapValues {
                it.value.associate { e -> e.key to e.value }
            }
    }

    fun bannerDataForStatusByBidsFromGrut(shard: Int, bids: List<Long>): List<BannerDataForStatus> {
        if (bids.isEmpty()) {
            return emptyList()
        }

        val grutBanners = bannerApi.getVersionedBannersByDirectId(bids)
        val grutBannersV2ByDirectBannerId = bannerV2Api.getBannersByDirectId(bids).associateBy { it.meta.directId }
        val campaignIds = grutBanners.asSequence().map { it.grutObject.meta.campaignId }.toSet()
        val campaignIdToType = campaignTypedRepository.getCampaignIdsWithType(shard, campaignIds)

        return grutBanners.map {
            convertToBannerDataForStatus(it, grutBannersV2ByDirectBannerId[it.grutObject.meta.directId], campaignIdToType)
        }
    }
}
