package ru.yandex.direct.core.aggregatedstatuses.logic;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatesEnum;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusWallet;
import ru.yandex.direct.core.entity.campaign.aggrstatus.CampaignDelayedOperation;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.campaign.service.WalletUtils;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;

import static java.math.RoundingMode.HALF_UP;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.BOUNDARY_NUMBER_OF_CONVERSIONS_PER_PERIOD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE;
import static ru.yandex.direct.currency.MoneyUtils.subtractPercentPrecisely;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * В классе вычисляются состояния, в которых находится кампания.
 */
public class CampaignStates implements StatesCalculator<CampaignStates.CampaignWithAdditionalDataForAggregatedStatus,
        CampaignStatesEnum> {
    /**
     * Предел, при котором мы предлагаем пользователям оплатить кампанию
     */
    private static final BigDecimal NEED_PAYMENT_BORDER = BigDecimal.valueOf(0.1);

    // костылик, для поддержки интерфейса, а интерфейс нужен чтобы обобщить классы для вычисления стейтов (косметика)
    public Collection<CampaignStatesEnum> calc(AggregatedStatusCampaign campaign, AggregatedStatusWallet wallet,
                                               Percent nds, Boolean isClientPessimized,
                                               @Nullable Long goalConversionsCount,
                                               boolean hasPromoExtensionRejected) {
        return calc(new CampaignWithAdditionalDataForAggregatedStatus(campaign, wallet, nds, isClientPessimized,
                goalConversionsCount, hasPromoExtensionRejected));
    }

    @Override
    public Collection<CampaignStatesEnum> calc(
            CampaignWithAdditionalDataForAggregatedStatus campaignWithAdditionalDataForAggregatedStatus) {
        AggregatedStatusCampaign campaign = campaignWithAdditionalDataForAggregatedStatus.getCampaign();
        AggregatedStatusWallet wallet = campaignWithAdditionalDataForAggregatedStatus.getWallet();
        Percent nds = campaignWithAdditionalDataForAggregatedStatus.getNds();

        List<CampaignStatesEnum> states = new ArrayList<>();
        if (nvl(campaign.getArchived(), false)) {
            states.addAll(archivedStates(campaign));
        } else if (CampaignDelayedOperation.ARC.equals(campaign.getDelayedOperation())) {
            states.add(CampaignStatesEnum.ARCHIVING);
        } else if (!nvl(campaign.getShowing(), true)) {
            states.add(CampaignStatesEnum.SUSPENDED);
        } else {
            if (cpmPriceCampaignNew(campaign)) {
                states.add(CampaignStatesEnum.CPM_PRICE_WAITING_FOR_APPROVE);
            } else if (cpmPriceCampaignNotApproved(campaign)) {
                states.add(CampaignStatesEnum.CPM_PRICE_NOT_APPROVED);
            } else if (cpmPriceCampaignIncorrect(campaign)) {
                states.add(CampaignStatesEnum.CPM_PRICE_INCORRECT);
            }

            if (nvl(campaign.getEmpty(), false) || CampaignStatusModerate.NEW.equals(campaign.getStatusModerate())) {
                states.add(CampaignStatesEnum.DRAFT);
            } else {
                BigDecimal sumTotal = campaignSumTotal(campaign, wallet);
                // для внутренних
                if (isInternalCampaign(campaign)) {
                    states.addAll(internalCampaignStates(campaign));
                    // для не внутренних
                } else if (!isValidId(campaign.getWalletId()) &&
                        //на кампании были деньги,но все потратилось
                        ((campaign.getSum().compareTo(Currencies.EPSILON) >= 0 &&
                                sumTotal.compareTo(Currencies.EPSILON) <= 0 && campaign.getOrderId() != 0)
                                //или на кампании вообще нет денег
                                || (campaign.getSum().compareTo(Currencies.EPSILON) <= 0
                                && campaign.getOrderId() == 0))
                ) {
                    states.add(CampaignStatesEnum.NO_MONEY);

                    if (Objects.nonNull(campaign.getSumToPay())
                            && campaign.getSumToPay().compareTo(Currencies.EPSILON) > 0) {
                        states.add(CampaignStatesEnum.AWAIT_PAYMENT);
                    }
                } else if (isValidId(campaign.getWalletId()) && sumTotal.compareTo(Currencies.EPSILON) <= 0) {
                    states.add(CampaignStatesEnum.NO_MONEY);

                    if (Objects.nonNull(wallet.getStatus()) && nvl(wallet.getStatus().getWaitingForPayment(), false)) {
                        states.add(CampaignStatesEnum.AWAIT_PAYMENT);
                    }
                } else {
                    states.add(CampaignStatesEnum.PAYED);
                }

                if (isCampaignNeedsNewPayment(campaign, sumTotal)) {
                    states.add(CampaignStatesEnum.NEED_PAYMENT);
                }
            }
        }

        // Временно убираем для всех кампаний: DIRECT-140177
        // boolean isPayForConversionStrategyData =
        //         isPayForConversionStrategyData(ifNotNull(campaign.getStrategy(), DbStrategy::getStrategyData));

        // if (isPayForConversionStrategyData &&
        //         hasLackOfFundsOnCampaignWithPayForConversion(campaign, wallet, nds)) {
        //     states.add(CampaignStatesEnum.PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_FUNDS);
        // }

        if (hasLackOfConversionOnCampaignWithPayForConversion(
                Optional.ofNullable(campaign.getStrategy()).map(DbStrategyBase::getStrategyData).orElse(null),
                campaignWithAdditionalDataForAggregatedStatus.isClientPessimized,
                campaignWithAdditionalDataForAggregatedStatus.goalConversionsCount)) {
            states.add(CampaignStatesEnum.PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_CONVERSION);
        }

        if (CampaignTypeKinds.ALLOW_DOMAIN_MONITORING.contains(campaign.getType())
                && nvl(campaign.getHasSiteMonitoring(), false)) {
            states.add(CampaignStatesEnum.DOMAIN_MONITORED);
        }

        if (campaignWithAdditionalDataForAggregatedStatus.hasPromoExtensionRejected) {
            states.add(CampaignStatesEnum.PROMO_EXTENSION_REJECTED);
        }

        return states;
    }

    @SuppressWarnings("unused")
    private boolean hasLackOfFundsOnCampaignWithPayForConversion(AggregatedStatusCampaign campaign,
                                                                 AggregatedStatusWallet wallet,
                                                                 Percent nds) {
        BigDecimal sumTotal = WalletUtils.calcSumTotalIncludingOverdraft(campaign.getSum(), campaign.getSumSpent(),
                ifNotNull(wallet, AggregatedStatusWallet::getSum),
                ifNotNull(wallet, AggregatedStatusWallet::getAutoOverdraftAddition));

        Currency currency = campaign.getCurrencyCode().getCurrency();

        return CampaignStrategyUtils.hasLackOfFundsOnCampaignWithPayForConversion(
                nds != null ? subtractPercentPrecisely(sumTotal, nds) : sumTotal,
                campaign.getStrategy().getStrategyData(),
                BigDecimal.valueOf(PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE),
                currency.getPayForConversionMinReservedSumDefaultValue());
    }

    private boolean hasLackOfConversionOnCampaignWithPayForConversion(
            @Nullable StrategyData strategyData,
            boolean isPessimizedClient,
            @Nullable Long goalConversionsCount) {
        if (strategyData == null
                || strategyData.getGoalId() == null
                || strategyData.getPayForConversion() == null
                || !strategyData.getPayForConversion()) {
            return false;
        }
        if (isPessimizedClient && goalConversionsCount != null) {
            return goalConversionsCount < BOUNDARY_NUMBER_OF_CONVERSIONS_PER_PERIOD;
        }
        return isPessimizedClient;
    }

    private List<CampaignStatesEnum> internalCampaignStates(AggregatedStatusCampaign campaign) {
        List<CampaignStatesEnum> states = new ArrayList<>();
        // Для дистрибуционных кампаний статусы в будущем будем считать по деньгам как для обычных кампаний
        // Сейчас их пропускаем, считая всегда активными (оплаченными). Для остальных считаем по юнитам.
        var unitsExhausted = campaign.getType() != CampaignType.INTERNAL_DISTRIB
                && Objects.nonNull(campaign.getRestrictionValue()) && Objects.nonNull(campaign.getSumSpentUnits())
                && BigDecimal.valueOf(campaign.getRestrictionValue() - campaign.getSumSpentUnits()).compareTo(Currencies.EPSILON) < 0;
        states.add(unitsExhausted ? CampaignStatesEnum.UNITS_EXHAUSTED : CampaignStatesEnum.PAYED);
        return states;
    }

    private boolean cpmPriceCampaignNew(AggregatedStatusCampaign campaign) {
        return isCpmPriceCampaign(campaign) && PriceFlightStatusApprove.NEW.equals(campaign.getFlightStatusApprove());
    }

    private boolean cpmPriceCampaignNotApproved(AggregatedStatusCampaign campaign) {
        return isCpmPriceCampaign(campaign) && PriceFlightStatusApprove.NO.equals(campaign.getFlightStatusApprove());
    }

    private boolean cpmPriceCampaignIncorrect(AggregatedStatusCampaign campaign) {
        boolean approved = PriceFlightStatusApprove.YES.equals(campaign.getFlightStatusApprove());
        boolean correct = PriceFlightStatusCorrect.YES.equals(campaign.getFlightStatusCorrect());
        return isCpmPriceCampaign(campaign) && approved && !correct;
    }

    private boolean isCpmPriceCampaign(AggregatedStatusCampaign campaign) {
        return CampaignType.CPM_PRICE.equals(campaign.getType());
    }

    private BigDecimal campaignSumTotal(AggregatedStatusCampaign campaign, AggregatedStatusWallet wallet) {
        BigDecimal sumTotal = campaign.getSum().subtract(campaign.getSumSpent());
        if (wallet != null) {
            if (sumTotal.compareTo(BigDecimal.ZERO) >= 0) {
                sumTotal = sumTotal.add(wallet.getSum());
            } else {
                //сумма на ОС уже учитывает перетраты на всех кампаниях(включая текущую).
                sumTotal = wallet.getSum();
            }
            // Если есть добавка по автоовердрафтам, учитываем её
            if (wallet.getAutoOverdraftAddition().compareTo(BigDecimal.ZERO) > 0) {
                sumTotal = sumTotal.add(wallet.getAutoOverdraftAddition());
            }
        }
        return sumTotal;
    }

    private List<CampaignStatesEnum> archivedStates(AggregatedStatusCampaign campaign) {
        List<CampaignStatesEnum> states = new ArrayList<>();
        if (CampaignDelayedOperation.UNARC.equals(campaign.getDelayedOperation())) {
            states.add(CampaignStatesEnum.UNARCHIVING);
        } else {
            states.add(CampaignStatesEnum.ARCHIVED);
            if (CurrencyCode.YND_FIXED.equals(campaign.getCurrencyCode())
                    && nvl(campaign.getCurrencyConverted(), false)) {
                states.add(CampaignStatesEnum.CANT_BE_UNARCHIVED);
            }
        }
        return states;
    }

    boolean isCampaignNeedsNewPayment(AggregatedStatusCampaign campaign, BigDecimal sumTotal) {
        return !isInternalCampaign(campaign) && !isValidId(campaign.getWalletId()) &&
                BigDecimal.ZERO.compareTo(campaign.getSumLast()) < 0 &&
                BigDecimal.ZERO.compareTo(sumTotal) < 0 &&
                NEED_PAYMENT_BORDER.compareTo(sumTotal.divide(campaign.getSumLast(), 3, HALF_UP))
                        >= 0;
    }

    // внутренние кампании, которые не оплачиваются деньгами, для части работает автопополнение
    boolean isInternalCampaign(AggregatedStatusCampaign campaign) {
        return campaign.getType() != null
                && CampaignTypeKinds.INTERNAL.contains(campaign.getType())
                && CampaignTypeKinds.NO_SEND_TO_BILLING.contains(campaign.getType());
    }

    public static class CampaignWithAdditionalDataForAggregatedStatus {
        private final AggregatedStatusCampaign campaign;
        private final AggregatedStatusWallet wallet;
        private final Percent nds;
        private final boolean isClientPessimized;
        private final Long goalConversionsCount;
        private final boolean hasPromoExtensionRejected;

        public CampaignWithAdditionalDataForAggregatedStatus(AggregatedStatusCampaign campaign,
                                                             AggregatedStatusWallet wallet,
                                                             Percent nds,
                                                             boolean isClientPessimized,
                                                             @Nullable Long goalConversionsCount,
                                                             boolean hasPromoExtensionRejected) {
            this.campaign = campaign;
            this.wallet = wallet;
            this.nds = nds;
            this.isClientPessimized = isClientPessimized;
            this.goalConversionsCount = goalConversionsCount;
            this.hasPromoExtensionRejected = hasPromoExtensionRejected;
        }

        public AggregatedStatusCampaign getCampaign() {
            return campaign;
        }

        public AggregatedStatusWallet getWallet() {
            return wallet;
        }

        public Percent getNds() {
            return nds;
        }
    }
}
