package ru.yandex.direct.core.aggregatedstatuses.logic;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AdStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusWallet;
import ru.yandex.direct.core.entity.campaign.aggrstatus.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatus;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatusInfo;
import ru.yandex.direct.utils.FunctionalUtils;

import static org.apache.commons.collections4.IterableUtils.matchesAny;
import static ru.yandex.direct.core.aggregatedstatuses.logic.StatusUtils.mergeStatuses;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.PAUSE_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_OK;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_OK;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_IS_PAUSED_BY_TIMETARGETING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus.status;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Статичные методы без сайдэффектов, бизнес логика для подсчета статуса объекта для отображения
 * пользователью, на основании собственного статуса объекта (на самом деле на основании содержимого
 * aggr_statuses_*.aggr_data), и собственных статусов его родительских объектов
 * В каких-то случаях может понадобится и сам объект (пример TimeTarget для кампании)
 */
public class EffectiveStatusCalculators {
    private static final Logger logger = LoggerFactory.getLogger(EffectiveStatusCalculators.class);

    private static final Set<GdSelfStatusReason> adgroupReasonMeaninglessOnAd = Set.of(
            GdSelfStatusReason.ADGROUP_ADS_SUSPENDED_BY_USER,
            GdSelfStatusReason.ADGROUP_ADS_WITH_WARNINGS,
            GdSelfStatusReason.ADGROUP_HAS_ADS_WITH_WARNINGS,
            GdSelfStatusReason.ADGROUP_HAS_NO_ADS_ELIGIBLE_FOR_SERVING,
            GdSelfStatusReason.ADGROUP_HAS_ADS_ON_MODERATION,
            GdSelfStatusReason.ADGROUP_ADS_REJECTED_ON_MODERATION,
            GdSelfStatusReason.ADGROUP_HAS_ADS_REJECTED_ON_MODERATION);

    private static final Set<GdSelfStatusReason> adgroupReasonsMeaninglessOnShowCondition = Set.of(
            GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_SUSPENDED_BY_USER,
            GdSelfStatusReason.ADGROUP_HAS_NO_SHOW_CONDITIONS_ELIGIBLE_FOR_SERVING,
            GdSelfStatusReason.ADGROUP_REJECTED_ON_MODERATION,
            GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_ON_MODERATION,
            GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_REJECTED_ON_MODERATION,
            GdSelfStatusReason.ADGROUP_HAS_SHOW_CONDITIONS_REJECTED_ON_MODERATION);

    private static final Set<GdSelfStatusReason> campaignReasonsMeaninglessOnAdGroup = Set.of(
            GdSelfStatusReason.CAMPAIGN_BL_PROCESSING_WITH_OLD_VERSION_SHOWN,
            GdSelfStatusReason.CAMPAIGN_BL_PROCESSING,
            GdSelfStatusReason.CAMPAIGN_BL_NOTHING_GENERATED,
            GdSelfStatusReason.CAMPAIGN_ALL_ADGROUPS_SUSPENDED_BY_USER,
            GdSelfStatusReason.CAMPAIGN_NO_ACTIVE_BANNERS,
            GdSelfStatusReason.CAMPAIGN_HAS_INACTIVE_BANNERS,
            GdSelfStatusReason.CAMPAIGN_HAS_NO_ADS_ELIGIBLE_FOR_SERVING,
            GdSelfStatusReason.CAMPAIGN_ON_MODERATION,
            GdSelfStatusReason.CAMPAIGN_ADGROUPS_PROCESSING,
            GdSelfStatusReason.CAMPAIGN_HAS_PROCESSING_ADGROUPS);

    public static final List<GdSelfStatusReason> GROUP_REASON_LOWER_DOWN_EVEN_IF_GROUP_ACTIVE = List.of(
            GdSelfStatusReason.ADGROUP_RARELY_SERVED,
            GdSelfStatusReason.ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN,
            GdSelfStatusReason.ADGROUP_HAS_RESTRICTED_GEO,
            GdSelfStatusReason.PROMO_EXTENSION_REJECTED);

    public static final List<GdSelfStatusReason> CAMPAIGN_REASON_LOWER_DOWN_EVEN_IF_CAMPAIGN_ACTIVE = List.of(
            GdSelfStatusReason.PROMO_EXTENSION_REJECTED);

    public static final Set<GdSelfStatusEnum> runStatuses = Set.of(RUN_OK, RUN_WARN, RUN_PROCESSING);

    private EffectiveStatusCalculators() {
        // noop
    }

    public static SelfStatus campaignEffectiveStatus(Instant currentInstant, AggregatedStatusCampaign campaign,
                                                     @Nullable Collection<AggregatedStatusCampaign> subCampaigns,
                                                     AggregatedStatusWallet wallet,
                                                     TimeTargetStatusInfo timeTargetStatus) {
        var today = currentInstant.atZone(MSK).toLocalDate();

        return campaignEffectiveStatus(today, campaign, subCampaigns, wallet, timeTargetStatus);
    }

    public static SelfStatus campaignEffectiveStatus(LocalDate today, AggregatedStatusCampaign campaign,
                                                     @Nullable Collection<AggregatedStatusCampaign> subCampaigns,
                                                     AggregatedStatusWallet wallet,
                                                     TimeTargetStatusInfo timeTargetStatus) {
        var campaignStatus = campaignEffectiveStatus(today, campaign, wallet, timeTargetStatus);
        if (CollectionUtils.isEmpty(subCampaigns) || campaignStatus == null) {
            return campaignStatus;
        }

        var subjectStatuses = StreamEx.of(subCampaigns)
                .nonNull()
                .map(subCampaign -> campaignEffectiveStatus(today, subCampaign, wallet, timeTargetStatus))
                .nonNull()
                .toList();

        var subjectReasons = StreamEx.of(subjectStatuses)
                .map(SelfStatus::getReasons)
                .nonNull()
                .toFlatCollection(Function.identity(), () -> EnumSet.noneOf(GdSelfStatusReason.class));

        var subjectRejectReasons = StreamEx.of(subjectStatuses)
                .flatMapToEntry(SelfStatus::getRejectReasons)
                .grouping(Collectors.flatMapping(StreamEx::of, Collectors.toSet()));

        if (matchesAny(subjectStatuses, EffectiveStatusCalculators::isActiveStatus)) {
            campaignStatus = setActive(campaignStatus);
        }
        if (!subjectRejectReasons.isEmpty() || subjectReasons.contains(GdSelfStatusReason.CAMPAIGN_HAS_NO_ADS_ELIGIBLE_FOR_SERVING)) {
            campaignStatus = addWarning(SelfStatus.copyAddRejectReasons(campaignStatus, subjectRejectReasons),
                    GdSelfStatusReason.UC_CAMPAIGN_SUBJECTS_REJECTED_ON_MODERATION);
        }
        if (subjectReasons.contains(GdSelfStatusReason.CAMPAIGN_BL_NOTHING_GENERATED)) {
            campaignStatus = addWarning(campaignStatus, GdSelfStatusReason.UC_CAMPAIGN_SUBJECTS_BL_NOTHING_GENERATED);
        }
        if (subjectReasons.contains(GdSelfStatusReason.CAMPAIGN_BL_PROCESSING)) {
            campaignStatus = addProcessingInfo(campaignStatus, GdSelfStatusReason.UC_CAMPAIGN_SUBJECTS_PROCESSING);
        }

        return campaignStatus;
    }

    public static SelfStatus campaignEffectiveStatus(Instant currentInstant, AggregatedStatusCampaign campaign,
                                                     AggregatedStatusWallet wallet,
                                                     TimeTargetStatusInfo timeTargetStatus) {
        LocalDate today = currentInstant.atZone(MSK).toLocalDate();

        return campaignEffectiveStatus(today, campaign, wallet, timeTargetStatus);
    }

    static SelfStatus campaignEffectiveStatus(LocalDate today, AggregatedStatusCampaign campaign,
                                              AggregatedStatusWallet wallet,
                                              TimeTargetStatusInfo timeTargetStatus) {
        AggregatedStatusCampaignData statusData = campaign.getAggregatedStatus();

        if (statusData == null) {
            return null;
        }

        GdSelfStatusEnum status = statusData.getStatusUnsafe();

        if (campaign.getSource() == CampaignSource.ZEN) {
            // Директ не знает реальный статус кампаний Дзена. Считаем, что они могут показываться.
            if (GdSelfStatusEnum.ARCHIVED.equals(status)) { // в административном порядке мы можем отправить кампанию в архив
                return status(GdSelfStatusEnum.ARCHIVED, GdSelfStatusReason.CAMPAIGN_IS_NOT_RECOVERABLE);
            } else {
                return status(RUN_OK, GdSelfStatusReason.ACTIVE);
            }
        }

        boolean campUnderWallet = isValidId(campaign.getWalletId());

        if (!GdSelfStatusEnum.ARCHIVED.equals(status)) {
            if (campaign.getFinishDate() != null
                    && campaign.getFinishDate().isBefore(today)) {
                return status(STOP_OK, GdSelfStatusReason.CAMPAIGN_IS_OVER);
            }
            if (campaign.getStrategy() != null
                    && campaign.getStrategy().getStrategyData().getFinish() != null
                    && campaign.getStrategy().getStrategyData().getFinish().isBefore(today)) {
                if (campaign.getStrategy().getStrategyData().getAutoProlongation() != null
                        && campaign.getStrategy().getStrategyData().getAutoProlongation() != 0) {
                    return status(RUN_WARN, GdSelfStatusReason.CAMPAIGN_STRATEGY_PERIOD_HAS_ENDED_AUTO_PROLONGATION);
                } else {
                    return status(STOP_OK, GdSelfStatusReason.CAMPAIGN_STRATEGY_PERIOD_HAS_ENDED);
                }
            }
        }

        if (!GdSelfStatusEnum.allRun().contains(status)) {
            return status(status, statusData.getReasons(), statusData.getRejectReasons());
        }

        if (campUnderWallet && wallet == null) {
            logger.error("No wallet found id: {} for campaign cid: {}", campaign.getWalletId(), campaign.getId());
            return status(status, statusData.getReasons(), statusData.getRejectReasons()); // fallback
        }

        if (campaign.getStartDate() != null && campaign.getStartDate().isAfter(today)) {
            return status(GdSelfStatusEnum.PAUSE_OK, GdSelfStatusReason.CAMPAIGN_IS_WAITING_START);
        } else if (campaign.getDayBudget() != null && campaign.getDayBudget().compareTo(BigDecimal.ZERO) > 0 &&
                campaign.getDayBudgetStopTime() != null &&
                campaign.getDayBudgetStopTime().toLocalDate().equals(today)) {
            return status(GdSelfStatusEnum.PAUSE_WARN, GdSelfStatusReason.CAMPAIGN_IS_PAUSED_BY_DAY_BUDGET);
        } else if (campUnderWallet &&
                wallet.getStatus().getBudgetLimitationStopTime() != null &&
                wallet.getStatus().getBudgetLimitationStopTime().toLocalDate().isEqual(today)) {
            return status(GdSelfStatusEnum.PAUSE_WARN, GdSelfStatusReason.CAMPAIGN_IS_PAUSED_BY_WALLET_DAY_BUDGET);
        } else if (timeTargetStatus != null && !TimeTargetStatus.ACTIVE.equals(timeTargetStatus.getStatus())) {
            return status(GdSelfStatusEnum.PAUSE_OK, CAMPAIGN_IS_PAUSED_BY_TIMETARGETING);
        }

        return status(status, statusData.getReasons(), statusData.getRejectReasons());
    }

    public static SelfStatus adGroupEffectiveStatus(Instant currentInstant, AggregatedStatusAdGroup adgroup,
                                                    AggregatedStatusCampaignData campaignData) {
        LocalDateTime now = currentInstant.atZone(MSK).toLocalDateTime();

        return adGroupEffectiveStatus(now, adgroup, campaignData);
    }

    public static SelfStatus adGroupEffectiveStatus(LocalDateTime now, AggregatedStatusAdGroup adgroup,
                                                    AggregatedStatusCampaignData campaignData) {
        AggregatedStatusAdGroupData adGroupData = adgroup.getAggregatedStatus();
        if (adGroupData == null || adGroupData.getStatus().isEmpty()) {
            return null;
        }

        SelfStatus adGroupStatus = adgroupTimeRelatedStatus(now, adgroup, adGroupData);

        if (campaignData == null || campaignData.getStatus().isEmpty()
                || GdSelfStatusEnum.ARCHIVED.equals(adGroupData.getStatusUnsafe())) {
            return adGroupStatus;
        }

        GdSelfStatusEnum campaignStatus = campaignData.getStatusUnsafe();
        if (GdSelfStatusEnum.ARCHIVED.equals(campaignStatus)) {
            return status(campaignStatus, campaignData.getReasons());
        }

        return mergeAdgroupAndCampaignStatuses(adGroupStatus, status(campaignStatus, campaignData.getReasons()));
    }

    private static SelfStatus adgroupTimeRelatedStatus(LocalDateTime now, AggregatedStatusAdGroup adgroup,
                                                       AggregatedStatusAdGroupData adGroupData) {
        boolean active = GdSelfStatusEnum.allRun().contains(adGroupData.getStatusUnsafe());
        if (active && adgroup.getStartTime() != null && adgroup.getStartTime().isAfter(now)) {
            return status(GdSelfStatusEnum.PAUSE_OK, GdSelfStatusReason.ADGROUP_IS_WAITING_START);
        } else if (active && adgroup.getFinishTime() != null && adgroup.getFinishTime().isBefore(now)) {
            return status(GdSelfStatusEnum.STOP_OK, GdSelfStatusReason.ADGROUP_IS_OVER);
        } else {
            return status(adGroupData.getStatusUnsafe(), adGroupData.getReasons(), adGroupData.getRejectReasons());
        }
    }

    public static SelfStatus adEffectiveStatus(AggregatedStatusAdData adData,
                                               AggregatedStatusAdGroupData adGroupData) {
        if (adData == null || adData.getStatus().isEmpty()) {
            return null;
        }

        final var allAdReasons = adData.getReasons();
        final var adGroupIsNotRun = adGroupData != null && adGroupData.getStatus().isPresent()
                && !runStatuses.contains(adGroupData.getStatus().get());

        /*
        Если группа не запущена, но при этом сам баннер промодерирован и готов к отправке в БК, то нам нужно
        убрать причину, которая говорит об этом, чтобы не сбивать пользователя с толку.
         */
        final var adReasons = adGroupIsNotRun && allAdReasons.contains(GdSelfStatusReason.AD_SENT_OR_READY_TO_BS)
                ? FunctionalUtils.filterList(allAdReasons, Predicate.not(Predicate.isEqual(GdSelfStatusReason.AD_SENT_OR_READY_TO_BS)))
                : allAdReasons;

        var adStatus = status(adData.getStatusUnsafe(), adReasons, adData.getRejectReasons());
        if (adGroupData == null || adGroupData.getStatus().isEmpty()
                || GdSelfStatusEnum.ARCHIVED.equals(adData.getStatusUnsafe())) {
            return adStatus;
        }

        if (adStatus.getStatus() == ON_MODERATION && !adStatus.getReasons().contains(AD_ON_MODERATION)) {
            adStatus = status(adStatus.getStatus(),
                    ImmutableList.<GdSelfStatusReason>builder()
                            .addAll(adData.getReasons())
                            .add(AD_ON_MODERATION)
                            .build(), adData.getRejectReasons());
        }

        GdSelfStatusEnum adGroupStatus = adGroupData.getStatusUnsafe();
        if (GdSelfStatusEnum.ARCHIVED.equals(adGroupStatus)) {
            return status(adGroupStatus, adGroupData.getReasons());
        }

        if (adData.getPlatform().equals(CampaignsPlatform.CONTEXT) || adData.getPlatform().equals(CampaignsPlatform.BOTH)) {
            Set<GdSelfStatusReason> allReasons = new HashSet<>(adStatus.getReasons());
            if (adData.getStates().contains(AdStatesEnum.HAS_SOCIAL_ADVERTISING_FLAG)) {
                allReasons.add(GdSelfStatusReason.AD_HAS_SOCIAL_ADVERTISING_FLAG);
            }
            if (adData.getStates().contains(AdStatesEnum.HAS_ASOCIAL_FLAG)) {
                allReasons.add(GdSelfStatusReason.AD_HAS_ASOCIAL_FLAG);
            }
            if (adData.getStates().contains(AdStatesEnum.HAS_UNFAMILY_FLAG)) {
                allReasons.add(GdSelfStatusReason.AD_HAS_UNFAMILY_FLAG);
            }
            if (adData.getStates().contains(AdStatesEnum.HAS_TRAGIC_FLAG)) {
                allReasons.add(GdSelfStatusReason.AD_HAS_TRAGIC_FLAG);
            }
            if (CollectionUtils.containsAny(allReasons, GdSelfStatusReason.AD_HAS_ASOCIAL_FLAG,
                    GdSelfStatusReason.AD_HAS_UNFAMILY_OR_TRAGIC_FLAG,
                    GdSelfStatusReason.AD_HAS_TRAGIC_FLAG,
                    GdSelfStatusReason.AD_HAS_UNFAMILY_FLAG)) {
                return mergeAdAndAdgroupStatuses(status(RUN_WARN, List.copyOf(allReasons)), status(adGroupStatus,
                        adGroupData.getReasons()));
            }
        }

        return mergeAdAndAdgroupStatuses(adStatus, status(adGroupStatus, adGroupData.getReasons()));
    }

    public static SelfStatus keywordEffectiveStatus(AggregatedStatusKeywordData keywordData,
                                                    AggregatedStatusAdGroupData adGroupData) {
        if (keywordData == null || keywordData.getStatus().isEmpty()) {
            return null;
        }

        var keywordStatus = status(keywordData.getStatusUnsafe(), keywordData.getReasons(),
                keywordData.getRejectReasons());
        if (adGroupData == null || adGroupData.getStatus().isEmpty()
                || GdSelfStatusEnum.ARCHIVED.equals(keywordData.getStatusUnsafe())) {
            return keywordStatus;
        }

        GdSelfStatusEnum adgroupStatus = adGroupData.getStatusUnsafe();
        if (GdSelfStatusEnum.ARCHIVED.equals(adgroupStatus)) {
            return status(adgroupStatus, adGroupData.getReasons());
        }

        return mergeShowConditionAndAdgroupStatuses(keywordStatus, status(adgroupStatus, adGroupData.getReasons()));
    }

    public static SelfStatus retargetingEffectiveStatus(AggregatedStatusRetargetingData retargetingData,
                                                        AggregatedStatusAdGroupData adGroupData) {
        if (retargetingData == null || retargetingData.getStatus().isEmpty()) {
            return null;
        }

        var retargetingStatus = status(retargetingData.getStatusUnsafe(), retargetingData.getReasons(),
                retargetingData.getRejectReasons());
        if (adGroupData == null || adGroupData.getStatus().isEmpty()) {
            return retargetingStatus;
        }

        GdSelfStatusEnum adGroupStatus = adGroupData.getStatusUnsafe();
        if (GdSelfStatusEnum.ARCHIVED.equals(adGroupStatus)) {
            return status(adGroupStatus, adGroupData.getReasons());
        }

        return mergeShowConditionAndAdgroupStatuses(retargetingStatus, status(adGroupStatus, adGroupData.getReasons()));
    }

    static SelfStatus mergeAdgroupAndCampaignStatuses(SelfStatus adgroupStatus, SelfStatus campaignStatus) {
        List<GdSelfStatusReason> campaignReasons = campaignStatus.getReasons().stream()
                .filter(r -> !campaignReasonsMeaninglessOnAdGroup.contains(r))
                .filter(r -> !ttOnStop(adgroupStatus, r))
                .collect(Collectors.toList());
        return mergeStatusesForCampaignSubjectsIfNotActive(
                adgroupStatus, status(campaignStatus.getStatus(), campaignReasons));
    }

    static SelfStatus mergeAdAndAdgroupStatuses(SelfStatus adStatus, SelfStatus adgroupStatus) {
        List<GdSelfStatusReason> adgroupReasons = adgroupStatus.getReasons().stream()
                .filter(r -> !adgroupReasonMeaninglessOnAd.contains(r))
                .filter(r -> !ttOnStop(adStatus, r))
                .collect(Collectors.toList());
        return mergeStatusesForAdGroupSubjectsIfNotActive(adStatus, status(adgroupStatus.getStatus(), adgroupReasons));
    }

    static SelfStatus mergeShowConditionAndAdgroupStatuses(SelfStatus showConditionStatus, SelfStatus adgroupStatus) {
        List<GdSelfStatusReason> adgroupReasons = adgroupStatus.getReasons().stream()
                .filter(r -> !ttOnStop(showConditionStatus, r))
                .filter(r -> !adgroupReasonsMeaninglessOnShowCondition.contains(r)).collect(Collectors.toList());
        return mergeStatusesForAdGroupSubjectsIfNotActive(showConditionStatus,
                status(adgroupStatus.getStatus(), adgroupReasons));
    }

    // если родитель показывается, то проблемы с его статусом не спускаем на подобъекты DIRECT-116341
    static SelfStatus mergeStatusesIfNotActive(SelfStatus subject, SelfStatus parent,
                                               List<GdSelfStatusReason> parentReasonsToLowerDownEvenIfParentActive) {
        if (!isActiveStatus(parent)) {
            return mergeStatuses(subject, parent);
        }
        var parentReasonsToLowerDown = filterList(parent.getReasons(), parentReasonsToLowerDownEvenIfParentActive::contains);
        return isEmpty(parentReasonsToLowerDown) ? subject :
                mergeStatuses(subject, status(parent.getStatus(), parentReasonsToLowerDown));
    }

    static SelfStatus mergeStatusesForCampaignSubjectsIfNotActive(SelfStatus subject, SelfStatus campaignStatus) {
        return mergeStatusesIfNotActive(subject, campaignStatus, CAMPAIGN_REASON_LOWER_DOWN_EVEN_IF_CAMPAIGN_ACTIVE);
    }

    static SelfStatus mergeStatusesForAdGroupSubjectsIfNotActive(SelfStatus subject, SelfStatus adgroupStatus) {
        return mergeStatusesIfNotActive(subject, adgroupStatus, GROUP_REASON_LOWER_DOWN_EVEN_IF_GROUP_ACTIVE);
    }

    static SelfStatus setActive(SelfStatus status) {
        switch (status.getStatus()) {
            case PAUSE_OK:
            case STOP_OK:
                return SelfStatus.copySetStatus(status, RUN_OK);
            case STOP_PROCESSING:
                return SelfStatus.copySetStatus(status, RUN_PROCESSING);
            case PAUSE_WARN:
            case PAUSE_CRIT:
            case STOP_WARN:
            case STOP_CRIT:
                return SelfStatus.copySetStatus(status, RUN_WARN);
            default:
                return status;
        }
    }

    static SelfStatus addWarning(SelfStatus status, GdSelfStatusReason reason) {
        switch (status.getStatus()) {
            case RUN_OK:
            case RUN_PROCESSING:
                return SelfStatus.copyAddReason(SelfStatus.copySetStatus(status, RUN_WARN), reason);
            case PAUSE_OK:
                return SelfStatus.copyAddSecondaryReason(SelfStatus.copySetStatus(status, PAUSE_WARN), reason);
            case STOP_OK:
            case STOP_PROCESSING:
                return SelfStatus.copyAddSecondaryReason(SelfStatus.copySetStatus(status, STOP_WARN), reason);
            default:
                return SelfStatus.copyAddSecondaryReason(status, reason);
        }
    }

    static SelfStatus addProcessingInfo(SelfStatus status, GdSelfStatusReason reason) {
        if (status.getStatus() == GdSelfStatusEnum.RUN_OK) {
            return status(RUN_PROCESSING, reason);
        }
        return SelfStatus.copyAddSecondaryReason(status, reason);
    }

    // для объектов в статусе STOP_OK не отображаем таймтаргет с кампании
    private static boolean ttOnStop(SelfStatus subject, GdSelfStatusReason reason) {
        return GdSelfStatusEnum.allStop().contains(subject.getStatus()) && CAMPAIGN_IS_PAUSED_BY_TIMETARGETING.equals(reason);
    }

    private static boolean isActiveStatus(SelfStatus status) {
        return GdSelfStatusEnum.allRun().contains(status.getStatus());
    }
}
