package ru.yandex.direct.core.aggregatedstatuses.logic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.aggregatedstatuses.logic.AdvancedCounters.AdvancedCounters;
import ru.yandex.direct.core.aggregatedstatuses.logic.AdvancedCounters.TotalCounter;
import ru.yandex.direct.core.aggregatedstatuses.logic.States.AdGroupStates;
import ru.yandex.direct.core.aggregatedstatuses.logic.States.CampaignStates;
import ru.yandex.direct.core.aggregatedstatuses.logic.Status.AdGroup.AdGroupStatuses;
import ru.yandex.direct.core.aggregatedstatuses.logic.Status.Campaign.CampaignStatuses;
import ru.yandex.direct.core.aggregatedstatuses.logic.Status.Status;
import ru.yandex.direct.core.aggregatedstatuses.logic.Status.Statuses;
import ru.yandex.direct.core.aggregatedstatuses.repository.ShowConditionsCounter;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AdStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.KeywordStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.RetargetingStatesEnum;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.aggregatedstatuses.logic.StatusUtils.sumStatuses;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.ARCHIVED;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.DRAFT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_OK;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_CRIT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_OK;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus.copyAddReasons;
import static ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus.status;
import static ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum.MODERATION;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

/**
 * Статичные методы без сайдэффектов, бизнес логика для подсчета собственного (без учета родительских объектов)
 * статуса объекта на основании его состояний (States) и счетчиков подъобъектов
 * Подробнее о системе статусов см. документацию: https://docs.yandex-team.ru/direct-dev/aggr-statuses/concept.html
 */
public class SelfStatusCalculators {
    private static final Logger logger = LoggerFactory.getLogger(SelfStatusCalculators.class);
    public static final Set<GdSelfStatusReason> ADGROUP_HAS_OBJECTS_ON_MODERATION =
            Set.of(GdSelfStatusReason.ADGROUP_HAS_ADS_ON_MODERATION,
                    GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_ON_MODERATION);

    public static final Map<AdStatesEnum, GdSelfStatusReason> WARN_STATES = ImmutableMap.<AdStatesEnum,
                    GdSelfStatusReason>builder()
            .put(AdStatesEnum.HAS_REJECTED_PLACEMENTS, GdSelfStatusReason.AD_HAS_REJECTED_PLACEMENTS)
            .put(AdStatesEnum.REJECTED_VCARD, GdSelfStatusReason.AD_VCARD_REJECTED_ON_MODERATION)
            .put(AdStatesEnum.REJECTED_VIDEO_ADDITION,
                    GdSelfStatusReason.AD_VIDEO_ADDITION_REJECTED_ON_MODERATION)
            .put(AdStatesEnum.HAS_REJECTED_CALLOUTS, GdSelfStatusReason.AD_HAS_REJECTED_CALLOUTS)
            .put(AdStatesEnum.HAS_REJECTED_SITELINKS, GdSelfStatusReason.AD_HAS_REJECTED_SITELINKS)
            .put(AdStatesEnum.REJECTED_IMAGE, GdSelfStatusReason.AD_IMAGE_REJECTED_ON_MODERATION)
            .put(AdStatesEnum.REJECTED_DISPLAY_HREF, GdSelfStatusReason.AD_DISPLAY_HREF_REJECTED_ON_MODERATION)
            .put(AdStatesEnum.REJECTED_TURBOLANDING, GdSelfStatusReason.AD_TURBOLANDING_REJECTED_ON_MODERATION)
            .put(AdStatesEnum.REJECTED_LOGO, GdSelfStatusReason.AD_LOGO_REJECTED_MODERATION)
            .put(AdStatesEnum.REJECTED_BUTTON, GdSelfStatusReason.AD_BUTTON_REJECTED_MODERATION)
            .put(AdStatesEnum.REJECTED_MULTICARD_SET, GdSelfStatusReason.AD_MULTICARD_SET_REJECTRED_MODERATION)
            .put(AdStatesEnum.HAS_PLACEMENTS_ON_OPERATOR_ACTIVATION,
                    GdSelfStatusReason.AD_HAS_PLACEMENTS_ON_OPERATOR_ACTIVATION)
            .build();

    private SelfStatusCalculators() {
        // noop
    }

    public static SelfStatus calcRelevanceMatchSelfStatus(Bid bid) {
        if (bid.getIsSuspended() != null && bid.getIsSuspended()) {
            return status(GdSelfStatusEnum.STOP_OK, GdSelfStatusReason.RELEVANCE_MATCH_SUSPENDED_BY_USER);
        } else if (bid.getIsDeleted() != null && bid.getIsSuspended()) {
            logger.error("Wrong state calculating status for deleted relevance match id: {}", bid.getId());
            return status(GdSelfStatusEnum.STOP_OK, GdSelfStatusReason.RELEVANCE_MATCH_SUSPENDED_BY_USER);
        }

        return status(GdSelfStatusEnum.RUN_OK);
    }

    public static SelfStatus calcRetargetingSelfStatus(Collection<RetargetingStatesEnum> states) {
        if (states.contains(RetargetingStatesEnum.SUSPENDED)) {
            return status(STOP_OK, GdSelfStatusReason.RETARGETING_SUSPENDED_BY_USER);
        }
        return status(GdSelfStatusEnum.RUN_OK);
    }

    public static SelfStatus calcKeywordSelfStatus(Collection<KeywordStatesEnum> states) {
        if (states.contains(KeywordStatesEnum.ARCHIVED)) {
            return status(ARCHIVED);
        } else if (states.contains(KeywordStatesEnum.DRAFT)) {
            return status(DRAFT);
        } else if (states.contains(KeywordStatesEnum.REJECTED)) {
            return status(STOP_CRIT, GdSelfStatusReason.REJECTED_ON_MODERATION);
        } else if (states.contains(KeywordStatesEnum.SUSPENDED)) {
            return status(STOP_OK, GdSelfStatusReason.KEYWORD_SUSPENDED_BY_USER);
        }
        return status(RUN_OK);
    }

    public static SelfStatus calcAdSelfStatus(Collection<AdStatesEnum> states,
                                              Map<ModerationDiagType,
                                              Set<Long>> rejectReasons) {
        SelfStatus status;
        boolean onModeration = states.contains(AdStatesEnum.MODERATION);
        List<GdSelfStatusReason> onModerationReasons = adAdditionsOnModerationReasons(states);
        if (states.contains(AdStatesEnum.ARCHIVED)) {
            status = status(ARCHIVED);
        } else if (states.contains(AdStatesEnum.DRAFT)) {
            status = onModeration
                    ? status(ON_MODERATION, GdSelfStatusReason.AD_ON_MODERATION)
                    : status(DRAFT);
        } else if (states.contains(AdStatesEnum.OBSOLETE)) {
            return status(states.contains(AdStatesEnum.ACTIVE_IN_BS) ? RUN_OK : STOP_OK);
        } else if (states.contains(AdStatesEnum.REJECTED) || states.contains(AdStatesEnum.ALL_PLACEMENTS_REJECTED)) {
            if (onModeration) {
                status = states.contains(AdStatesEnum.ACTIVE_IN_BS)
                        ? status(RUN_WARN,
                            List.of(GdSelfStatusReason.AD_ON_MODERATION,
                                    GdSelfStatusReason.AD_REJECTED_BUT_PREVIOUS_VERSION_SHOWN),
                            rejectReasons) // RUN_CRIT?
                        : status(STOP_CRIT, GdSelfStatusReason.AD_ON_MODERATION);
            } else {
                status = states.contains(AdStatesEnum.ACTIVE_IN_BS)
                        ? status(RUN_WARN, GdSelfStatusReason.AD_REJECTED_BUT_PREVIOUS_VERSION_SHOWN, rejectReasons)
                        // RUN_CRIT?
                        : status(STOP_CRIT, GdSelfStatusReason.REJECTED_ON_MODERATION, rejectReasons);
            }
        } else if (states.contains(AdStatesEnum.SUSPENDED)) {
            status = status(STOP_OK, onModeration
                    ? GdSelfStatusReason.AD_ON_MODERATION : GdSelfStatusReason.AD_SUSPENDED_BY_USER);
        } else if (states.contains(AdStatesEnum.SUSPENDED_BY_MONITORING)) {
            List<GdSelfStatusReason> reasons = new ArrayList<>();
            reasons.add(GdSelfStatusReason.OFF_BY_MONITORING);
            if (onModeration) {
                reasons.add(GdSelfStatusReason.AD_ON_MODERATION);
            }
            status = status(STOP_OK, reasons);
        } else if (states.contains(AdStatesEnum.HAS_PLACEMENTS_ON_MODERATION)
                && !states.contains(AdStatesEnum.HAS_ACCEPTED_PLACEMENTS)) {
            status = states.contains(AdStatesEnum.ACTIVE_IN_BS)
                    ? status(RUN_OK, GdSelfStatusReason.AD_ON_MODERATION_PREVIOUS_VERSION_SHOWN)
                    : status(DRAFT, GdSelfStatusReason.AD_HAS_PLACEMENTS_ON_OPERATOR_MODERATION);
        } else if (onModeration) {
            status = states.contains(AdStatesEnum.ACTIVE_IN_BS)
                    ? status(RUN_OK, GdSelfStatusReason.AD_ON_MODERATION_PREVIOUS_VERSION_SHOWN)
                    : status(ON_MODERATION, GdSelfStatusReason.AD_ON_MODERATION);
        } else if (states.contains(AdStatesEnum.PLACEMENTS_REQUIRED)) {
            status = status(STOP_CRIT, GdSelfStatusReason.AD_NO_SUITABLE_PLACEMENTS_SELECTED);
        } else if (!Sets.intersection(WARN_STATES.keySet(), new HashSet<>(states)).isEmpty()) {
            status = status(RUN_WARN, emptyList());
        } else if (!onModerationReasons.isEmpty()) {
            status = status(GdSelfStatusEnum.RUN_OK, onModerationReasons);
        } else if (states.contains(AdStatesEnum.ACTIVE_IN_BS)) {
            status = status(GdSelfStatusEnum.RUN_OK);
        } else if (states.contains(AdStatesEnum.READY_TO_BS)) {
            status = status(RUN_PROCESSING, GdSelfStatusReason.AD_SENT_OR_READY_TO_BS);
        } else {
            // Для объявления RUN_PROCESSING здесь означает, что из той информации, что есть в самом объявлении,
            // следует, что показы должны идти, но они почему-то не идут - а причина станет известной на стадии
            // вычисления статуса группы или кампании (например, идет генерация в BannerLand).
            // Этот статус никогда не должен стать эффективным статусом объявления, так как это состояние означает,
            // что или группа, или кампания будут иметь STOP_* статус, который при вычислении эффективного статуса
            // "спустится" на это объявление.
            // Здесь не используется более "честный" STOP_* статус, так как при вычислении статуса группы нам
            // важно понимать, "запускаемое" ли объявление "в отрыве" от группы (в данном случае - оно таковое),
            // но при этом важно отличить и то, может ли оно быть запущено на самом деле - поэтому здесь эта развилка.
            status = status(RUN_PROCESSING);
        }

        if (RUN_WARN.equals(status.getStatus())) {
            status = copyAddReasons(status, mapAndFilterList(states, WARN_STATES::get, Objects::nonNull));
        }

        return status;
    }

    private static List<GdSelfStatusReason> adAdditionsOnModerationReasons(Collection<AdStatesEnum> states) {
        List<GdSelfStatusReason> reasons = new ArrayList<>();

        if (states.contains(AdStatesEnum.CALLOUTS_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_CALLOUTS_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.VCARD_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_VCARD_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.SITELINKS_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_SITELINKS_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.LOGO_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_LOGO_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.BUTTON_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_BUTTON_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.MULTICARD_SET_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_MULTICARD_SET_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.IMAGE_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_IMAGE_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.DISPLAY_HREF_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_DISPLAY_HREF_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.TURBOLANDING_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_TURBOLANDING_ON_MODERATION);
        }
        if (states.contains(AdStatesEnum.VIDEO_ADDITION_ON_MODERATION)) {
            reasons.add(GdSelfStatusReason.AD_VIDEO_ADDITION_ON_MODERATION);
        }

        return reasons;
    }

    public static SelfStatus calcAdGroupSelfStatus(AggregatedStatusAdGroup adgroup,
                                                   Collection<AdGroupStatesEnum> states,
                                                   Map<ModerationDiagType, Set<Long>> rejectReasons,
                                                   AdGroupCounters counters,
                                                   boolean hasInterestShowCondition,
                                                   boolean hasBannerGeoLegalFlag,
                                                   ShowConditionsCounter showConditionsCounter,
                                                   boolean useNewCalculator) {
        var selfStatus = useNewCalculator ? calcAdGroupSelfStatus1(
                adgroup, states, counters, hasInterestShowCondition, hasBannerGeoLegalFlag, showConditionsCounter
        ) : calcAdGroupSelfStatus0(
                adgroup, states, counters, hasInterestShowCondition, hasBannerGeoLegalFlag, showConditionsCounter
        );

        return SelfStatus.status(selfStatus.getStatus(), selfStatus.getReasons(), rejectReasons);
    }

    private static SelfStatus calcAdGroupSelfStatus1(AggregatedStatusAdGroup adgroup,
                                                     Collection<AdGroupStatesEnum> states,
                                                     AdGroupCounters counters,
                                                     boolean hasInterestShowCondition,
                                                     boolean hasBannerGeoLegalFlag,
                                                     ShowConditionsCounter showConditionsCounter) {
        int activeInterests = hasInterestShowCondition ? 1 : 0; // always active
        int internalShowConditions = AdGroupType.INTERNAL.equals(adgroup.getType()) ? 1 : 0; // always active

        final var adGroupStates = new AdGroupStates(states, hasBannerGeoLegalFlag);
        final var totalCounter = new TotalCounter(0, counters.getAdsTotal(), 1,
                counters.getKeywordsTotal(), counters.getRetargetingsTotal(), activeInterests, internalShowConditions,
                showConditionsCounter.getTotal());

        final var advancedCounters = new AdvancedCounters(totalCounter, counters, showConditionsCounter,
                activeInterests, internalShowConditions);

        final Statuses adGroupStatuses = new AdGroupStatuses(adGroupStates, totalCounter, advancedCounters);
        final var status = getSelfStatus(adGroupStatuses);

        return status.orElseGet(() -> status(GdSelfStatusEnum.RUN_OK));
    }

    private static SelfStatus calcAdGroupSelfStatus0(AggregatedStatusAdGroup adgroup,
                                                     Collection<AdGroupStatesEnum> states,
                                                     AdGroupCounters counters,
                                                     boolean hasInterestShowCondition,
                                                     boolean hasBannerGeoLegalFlag,
                                                     ShowConditionsCounter showConditionsCounter) {
        int activeInterests = hasInterestShowCondition ? 1 : 0; // always active
        int internalShowConditions = AdGroupType.INTERNAL.equals(adgroup.getType()) ? 1 : 0; // always active
        int adsTotal = counters.getAdsTotal();
        int keywordsTotal = counters.getKeywordsTotal();
        int retargetingsTotal = counters.getRetargetingsTotal();
        int showConditionsTotal = keywordsTotal + retargetingsTotal + activeInterests + internalShowConditions
                + showConditionsCounter.getTotal();

        int draftAds = counters.getAdStatuses().getOrDefault(DRAFT, 0);
        int suspendedAds = counters.getAdStatuses().getOrDefault(STOP_OK, 0);
        int rejectedAds = counters.getAdStatuses().getOrDefault(STOP_CRIT, 0);
        int archivedAds = counters.getAdStatuses().getOrDefault(ARCHIVED, 0);
        int onModerationAds = counters.getAdStatuses().getOrDefault(ON_MODERATION, 0);
        int nonArchivedAds = adsTotal - archivedAds;
        int runnableAds = nonArchivedAds - suspendedAds - draftAds - onModerationAds;
        int activeAds = sumStatuses(GdSelfStatusEnum.allRun(), counters.getAdStatuses());
        int processingAds = sumStatuses(GdSelfStatusEnum.allProcessing(), counters.getAdStatuses());
        int okAds = sumStatuses(GdSelfStatusEnum.allOk(), counters.getAdStatuses());
        int warnedAds = counters.getAdStatuses().getOrDefault(RUN_WARN, 0) +
                counters.getAdStatuses().getOrDefault(STOP_WARN, 0);

        boolean hasAdsOnModeration = onModerationAds > 0;
        boolean allAdsSuspended = suspendedAds > 0 && suspendedAds == nonArchivedAds;
        boolean hasRejectedAds = rejectedAds > 0;
        boolean allAdsRejected = hasRejectedAds && rejectedAds == nonArchivedAds;
        boolean hasActiveAds = activeAds > 0;
        boolean allAdsActive = hasActiveAds && activeAds == runnableAds;
        boolean allDraftAds = draftAds > 0 && nonArchivedAds == draftAds;
        boolean hasProcessingAds = processingAds > 0;
        boolean allAdsProcessing = hasProcessingAds && processingAds == runnableAds;
        boolean hasAdsWithWarnings = warnedAds > 0;
        boolean allAdsWarned = hasAdsWithWarnings && warnedAds == nonArchivedAds;
        boolean allAdsOk = okAds == adsTotal;

        int archivedKeywords = counters.getKeywordStatuses().getOrDefault(ARCHIVED, 0);
        int suspendedKeywords = counters.getKeywordStatuses().getOrDefault(STOP_OK, 0);
        int rejectedKeywords = counters.getKeywordStatuses().getOrDefault(STOP_CRIT, 0);
        int draftKeywords = counters.getKeywordStatuses().getOrDefault(DRAFT, 0);
        int nonArchivedKeywords = keywordsTotal - archivedKeywords;
        int runnableKeywords = nonArchivedKeywords - suspendedKeywords - draftKeywords;
        int activeKeywords = sumStatuses(GdSelfStatusEnum.allRun(), counters.getKeywordStatuses());
        int warnedKeywords = counters.getKeywordStatuses().getOrDefault(RUN_WARN, 0) +
                counters.getKeywordStatuses().getOrDefault(STOP_WARN, 0);

        int suspendedRetargetings = counters.getRetargetingStatuses().getOrDefault(STOP_OK, 0);
        int runnableRetargetings = retargetingsTotal - suspendedRetargetings;
        int activeRetargetings = sumStatuses(GdSelfStatusEnum.allRun(), counters.getRetargetingStatuses());

        int runnableShowConditions = runnableKeywords + runnableRetargetings + activeInterests + internalShowConditions
                + showConditionsCounter.getNonSuspended();
        int suspendedShowConditions = suspendedKeywords + suspendedRetargetings + showConditionsCounter.getSuspended();
        int nonArchivedShowConditions = nonArchivedKeywords + retargetingsTotal + internalShowConditions
                + showConditionsCounter.getTotal();
        int activeShowConditions = activeKeywords + activeRetargetings + activeInterests + internalShowConditions
                + showConditionsCounter.getNonSuspended();
        int rejectedShowConditions = rejectedKeywords;
        int warnedShowConditions = warnedKeywords;

        boolean hasRejectedShowConditions = rejectedShowConditions > 0;
        boolean hasDraftKeywords = draftKeywords > 0;
        boolean hasShowConditionsWithWarnings = warnedShowConditions > 0;
        boolean hasActiveShowConditions = activeShowConditions > 0;
        boolean allShowConditionsRejected = hasRejectedShowConditions
                && rejectedShowConditions == nonArchivedShowConditions;
        boolean allShowConditionsSuspended = suspendedShowConditions > 0
                && suspendedShowConditions == nonArchivedShowConditions;
        boolean allShowConditionsActive = hasActiveShowConditions && activeShowConditions == runnableShowConditions;

        boolean hasAdsDraftOnModeration = counters.getAdStates()
                .getOrDefault(AdStatesEnum.DRAFT_ON_MODERATION, 0) > 0;

        boolean hasRestrictedGeo = states.contains(AdGroupStatesEnum.HAS_RESTRICTED_GEO);
        boolean hasNoEffectiveGeo = states.contains(AdGroupStatesEnum.HAS_NO_EFFECTIVE_GEO);

        boolean statusBlGeneratedIsProcessing = states.contains(AdGroupStatesEnum.STATUS_BL_GENERATED_PROCESSING);
        boolean statusBlGeneratedNothingGenerated =
                states.contains(AdGroupStatesEnum.STATUS_BL_GENERATED_NOTHING_GENERATED);
        boolean adgroupOnModeration = states.contains(MODERATION);

        List<GdSelfStatusReason> adgroupIsolatedReasons = new ArrayList<>();
        if (runnableShowConditions > 0) {
            // смотрим на статус модерации группы, только если в ней есть условия показа и баннеры
            if (states.contains(AdGroupStatesEnum.REJECTED)) {
                adgroupIsolatedReasons.add(GdSelfStatusReason.ADGROUP_REJECTED_ON_MODERATION);
            } else if (adgroupOnModeration && adsTotal > 0) {
                adgroupIsolatedReasons.add(getReasonAdgroupShowConditionsOnModeration(hasDraftKeywords,
                        hasRejectedShowConditions, hasActiveShowConditions));
            }
        }

        if (hasAdsDraftOnModeration) {
            adgroupIsolatedReasons.add(GdSelfStatusReason.ADGROUP_HAS_ADS_ON_MODERATION);
        }

        if (archivedAds > 0 && archivedAds == adsTotal) {
            return status(ARCHIVED);
        } else if (states.contains(AdGroupStatesEnum.DRAFT)
                || adsTotal < 1
                || showConditionsTotal < 1
                || allDraftAds) {
            if (statusBlGeneratedIsProcessing) {
                adgroupIsolatedReasons.add(GdSelfStatusReason.ADGROUP_BL_PROCESSING);
            }
            if (showConditionsTotal < 1) {
                adgroupIsolatedReasons.add(GdSelfStatusReason.ADGROUP_HAS_NO_SHOW_CONDITIONS_ELIGIBLE_FOR_SERVING);
            }
            return !adgroupIsolatedReasons.isEmpty()
                    ? status(DRAFT, adgroupIsolatedReasons)
                    : status(DRAFT);
        } else if (allAdsSuspended || allShowConditionsSuspended) {
            var reasons = new ArrayList<GdSelfStatusReason>();
            if (allAdsSuspended) {
                reasons.add(GdSelfStatusReason.ADGROUP_ADS_SUSPENDED_BY_USER);
            }
            if (allShowConditionsSuspended) {
                reasons.add(GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_SUSPENDED_BY_USER);
            }
            return status(STOP_OK, reasons);
        } else if (statusBlGeneratedIsProcessing && allAdsOk) {
            if (allAdsProcessing || !hasActiveAds || !hasActiveShowConditions) {
                return status(STOP_PROCESSING, GdSelfStatusReason.ADGROUP_BL_PROCESSING);
            }
            if (hasProcessingAds) {
                return status(RUN_PROCESSING, GdSelfStatusReason.ADGROUP_BL_PROCESSING);
            }
            return status(RUN_PROCESSING, GdSelfStatusReason.ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN);
            // имеет смысл только если группа не пустая, т.к. у новых всегда будет statusBlGenerated = No
        } else if (statusBlGeneratedNothingGenerated && hasActiveAds) {
            return status(STOP_CRIT, GdSelfStatusReason.ADGROUP_BL_NOTHING_GENERATED);
        } else if (hasRestrictedGeo) {
            return status(RUN_WARN, GdSelfStatusReason.ADGROUP_HAS_RESTRICTED_GEO);
        } else if (hasNoEffectiveGeo) {
            return status(STOP_CRIT,
                    hasBannerGeoLegalFlag ? GdSelfStatusReason.ADGROUP_NEED_DOCUMENTS_FOR_MODERATION
                            : GdSelfStatusReason.ADGROUP_HAS_NO_EFFECTIVE_GEO);
        } else if (allShowConditionsActive && allAdsActive && !hasAdsWithWarnings && !hasShowConditionsWithWarnings
                && !adgroupOnModeration) {
            return status(RUN_OK);
        } else if (hasActiveAds && hasActiveShowConditions) {
            List<GdSelfStatusReason> reasons = new ArrayList<>();
            if (hasRejectedAds) {
                reasons.add(allAdsRejected
                        ? GdSelfStatusReason.ADGROUP_ADS_REJECTED_ON_MODERATION
                        : GdSelfStatusReason.ADGROUP_HAS_ADS_REJECTED_ON_MODERATION);
            } else if (hasAdsWithWarnings) {
                reasons.add(allAdsWarned
                        ? GdSelfStatusReason.ADGROUP_ADS_WITH_WARNINGS
                        : GdSelfStatusReason.ADGROUP_HAS_ADS_WITH_WARNINGS);
            }
            if (hasRejectedShowConditions) {
                reasons.add(allShowConditionsRejected
                        ? GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_REJECTED_ON_MODERATION
                        : GdSelfStatusReason.ADGROUP_HAS_SHOW_CONDITIONS_REJECTED_ON_MODERATION);
            } else if (hasShowConditionsWithWarnings) {
                reasons.add(GdSelfStatusReason.ADGROUP_HAS_SHOW_CONDITIONS_WITH_WARNINGS);
            } else if (!adgroupIsolatedReasons.isEmpty()) {
                reasons.addAll(adgroupIsolatedReasons);
            }
            return reasons.isEmpty() ? status(RUN_OK) : status(RUN_WARN, reasons);
        } else if ((hasActiveAds || draftAds > 0) && adgroupOnModeration) {
            // https://st.yandex-team.ru/DIRECT-130094#5fa18fe985e3ae5592fd8972
            return status(ON_MODERATION, getReasonAdgroupShowConditionsOnModeration(hasDraftKeywords,
                    hasRejectedShowConditions, hasActiveShowConditions));
        } else if (hasAdsOnModeration) {
            return status(ON_MODERATION, GdSelfStatusReason.ADGROUP_HAS_ADS_ON_MODERATION);
        } else if (runnableAds > 0 && !hasActiveAds) {
            // есть не остановленные и не архивные объявления, но при этом нет активных
            // если все банеры на модерации будет STOP_CRIT, что может быть не совсем  правильно, но не лечится
            // для текущих стейтов (или стейты тоже надо через else if между собой пересекать).
            return status(STOP_CRIT, unshiftAndCopyReasons(adgroupIsolatedReasons,
                    GdSelfStatusReason.ADGROUP_HAS_NO_ADS_ELIGIBLE_FOR_SERVING));
        } else if (runnableShowConditions > 0 && !hasActiveShowConditions) {
            // есть не остановленные и не архивные условия показа, но при этом нет активных
            return status(STOP_CRIT, !adgroupIsolatedReasons.isEmpty()
                    ? adgroupIsolatedReasons
                    : List.of(GdSelfStatusReason.ADGROUP_HAS_NO_SHOW_CONDITIONS_ELIGIBLE_FOR_SERVING));
        } else if (draftAds > 0 && suspendedAds > 0) { // overwise one of the previous will work
            return !adgroupIsolatedReasons.isEmpty()
                    ? status(DRAFT, adgroupIsolatedReasons)
                    : status(DRAFT);
        }

        return status(GdSelfStatusEnum.RUN_OK);
    }

    private static GdSelfStatusReason getReasonAdgroupShowConditionsOnModeration(boolean hasDraftKeywords,
                                                                                 boolean hasRejectedShowConditions,
                                                                                 boolean hasActiveShowConditions) {
        if (hasDraftKeywords && (hasActiveShowConditions || hasRejectedShowConditions)) {
            return GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_PARTLY_ON_MODERATION;
        } else {
            return GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_ON_MODERATION;
        }
    }

    @Nullable
    public static SelfStatus calcCampaignSelfStatus(Long cid, boolean hasWallet,
                                                    Collection<CampaignStatesEnum> states,
                                                    Map<ModerationDiagType, Set<Long>> rejectReasons,
                                                    CampaignCounters counters,
                                                    @Nullable Boolean isDraftApproveAllowed,
                                                    boolean useNewCalculator) {
        var status = useNewCalculator
                ? calcCampaignSelfStatus1(cid, hasWallet, states, counters, isDraftApproveAllowed)
                : calcCampaignSelfStatus0(cid, hasWallet, states, counters, isDraftApproveAllowed);

        if (status != null) {
            return SelfStatus.copyAddRejectReasons(status, rejectReasons);
        }
        return null;
    }

    @Nullable
    private static SelfStatus calcCampaignSelfStatus1(Long cid, boolean hasWallet,
                                                      Collection<CampaignStatesEnum> states,
                                                      CampaignCounters counters,
                                                      @Nullable Boolean isDraftApproveAllowed) {
        final var totalCounter = new TotalCounter(1, 0, counters.getAdgroupsTotal(), 0, 0, 0, 0, 0);
        final var campaignAdvancedCounters = new AdvancedCounters(totalCounter, counters);
        final var campaignStates = new CampaignStates(states, isDraftApproveAllowed, hasWallet);
        final var campaignStatuses = new CampaignStatuses(campaignStates, totalCounter, campaignAdvancedCounters);
        final var status = getSelfStatus(campaignStatuses);

        if (status.isEmpty() && campaignStates.isUnarchiving()) {
            return null;
        } else if (status.isEmpty()) {
            logger.error("Error while calculating selfStatus for cid: {}", cid);
            return null; // какое-то не предусмотренное состояние
        }

        return status.get();
    }

    private static Optional<SelfStatus> getSelfStatus(Statuses statuses) {
        final var filteredStatuses = statuses
                .possibleStatuses()
                .stream()
                .filter(Status::isValid)
                .map(Status::selfStatus)
                .collect(Collectors.toList());

        if (filteredStatuses.size() > 1) {
            logger.warn("Ambiguous state: {}.", filteredStatuses);
        }

        if (filteredStatuses.isEmpty()) {
            return Optional.empty();
        }

        return Optional.of(filteredStatuses.get(0));
    }

    @Nullable
    private static SelfStatus calcCampaignSelfStatus0(Long cid, boolean hasWallet,
                                                      Collection<CampaignStatesEnum> states,
                                                      CampaignCounters counters,
                                                      @Nullable Boolean isDraftApproveAllowed) {
        int adgroupsTotal = counters.getAdgroupsTotal();
        int archivedAdGroups = counters.getStatuses().getOrDefault(ARCHIVED, 0);
        int suspendedAdGroups = counters.getStatuses().getOrDefault(STOP_OK, 0);
        int warnedAdGroups = counters.getStatuses().getOrDefault(RUN_WARN, 0);
        int draftAdGroups = counters.getStatuses().getOrDefault(DRAFT, 0);
        int onModerationAdGroups = counters.getStatuses().getOrDefault(ON_MODERATION, 0);
        int activeAdgroups = sumStatuses(GdSelfStatusEnum.allRun(), counters.getStatuses());
        int okAdgroups = sumStatuses(GdSelfStatusEnum.allOk(), counters.getStatuses());
        int processingAdGroups = sumStatuses(GdSelfStatusEnum.allProcessing(), counters.getStatuses());
        int nonArchivedAdgroups = adgroupsTotal - archivedAdGroups;
        int runnableAdgroups = nonArchivedAdgroups - suspendedAdGroups - draftAdGroups;

        boolean allAdGroupsArchived = adgroupsTotal > 0 && archivedAdGroups == adgroupsTotal;
        boolean allAdgroupsSuspended = suspendedAdGroups > 0 && suspendedAdGroups == nonArchivedAdgroups;
        boolean hasAdGroupsOnModeration = onModerationAdGroups > 0;
        boolean hasActiveAdgroups = activeAdgroups > 0;
        boolean hasWarnedAdGroups = warnedAdGroups > 0;
        boolean hasProcessingAdGroups = processingAdGroups > 0;
        boolean allAdgroupsActive = hasActiveAdgroups && activeAdgroups == runnableAdgroups;
        boolean allAdgroupsOk = okAdgroups == adgroupsTotal;
        boolean allAdgroupsDraft = draftAdGroups > 0 && draftAdGroups == nonArchivedAdgroups;
        boolean allAdgroupsDraftOrSuspended = (suspendedAdGroups > 0 || draftAdGroups > 0)
                && (suspendedAdGroups + draftAdGroups) == nonArchivedAdgroups;
        boolean allAdGroupsProcessing = processingAdGroups == runnableAdgroups;

        boolean hasPromoExtensionRejected = states.contains(CampaignStatesEnum.PROMO_EXTENSION_REJECTED);

        boolean campaignPayed = states.contains(CampaignStatesEnum.PAYED);

        boolean hasDraftOnModerationAdGroups = counters.getStates()
                .getOrDefault(AdGroupStatesEnum.HAS_DRAFT_ON_MODERATION_ADS, 0) > 0;

        int blNothingGeneratedAdGroups = counters.getStates()
                .getOrDefault(AdGroupStatesEnum.STATUS_BL_GENERATED_NOTHING_GENERATED, 0);
        boolean allStoppedByNothingGenerated = blNothingGeneratedAdGroups != 0
                && counters.getStatuses().getOrDefault(STOP_CRIT, 0) == blNothingGeneratedAdGroups;

        int blGeneratedProcessingAdGroups = counters.getStates()
                .getOrDefault(AdGroupStatesEnum.STATUS_BL_GENERATED_PROCESSING, 0);
        boolean allStoppedByBlProcessing = blGeneratedProcessingAdGroups != 0
                && counters.getStatuses().getOrDefault(STOP_PROCESSING, 0) == blGeneratedProcessingAdGroups;

        GdSelfStatusReason cpmPriceReason = cpmPriceReasons(states);

        List<GdSelfStatusReason> campaignIsolatedReasons = new ArrayList<>();
        if (hasDraftOnModerationAdGroups || hasAdGroupsOnModeration) {
            if (hasActiveAdgroups) {
                campaignIsolatedReasons.add(GdSelfStatusReason.CAMPAIGN_PARTLY_ON_MODERATION);
            } else {
                // Если прайсовая кампания на модерации - то всегда сообщаем статус бронирования первым пунктом
                // чтобы в интерфейсе было например "На Модерации. Бронирование подтверждено".`
                ifNotNull(cpmPriceReason, campaignIsolatedReasons::add);
                campaignIsolatedReasons.add(GdSelfStatusReason.CAMPAIGN_ON_MODERATION);
            }
        }
        if (!campaignIsolatedReasons.contains(GdSelfStatusReason.CAMPAIGN_ON_MODERATION)
                && cpmPriceReason != null) {
            // В прайсовых для Черновика проставляем статус бронирования во всех случаях кроме одного:
            // Статус бронирования "Ожидает бронирования" но на пакете нет возможности "Предбронировать черновик"
            //noinspection ConstantConditions
            if ((cpmPriceReason != GdSelfStatusReason.CPM_PRICE_WAITING_FOR_APPROVE)
                    // можно unbox т.к. для любой прайсовой кампании должен быть определён isDraftApproveAllowed
                    || isDraftApproveAllowed) {
                campaignIsolatedReasons = unshiftAndCopyReasons(campaignIsolatedReasons, cpmPriceReason);
            }
        }

        if (states.contains(CampaignStatesEnum.ARCHIVED)
                || states.contains(CampaignStatesEnum.ARCHIVING)
                || allAdGroupsArchived) {
            return states.contains(CampaignStatesEnum.CANT_BE_UNARCHIVED)
                    ? status(ARCHIVED, GdSelfStatusReason.CAMPAIGN_IS_NOT_RECOVERABLE)
                    : status(ARCHIVED);
        } else if (states.contains(CampaignStatesEnum.DRAFT)) {
            return getCampaignSelfStatus(campaignIsolatedReasons);
        } else if (adgroupsTotal < 1) {
            return status(DRAFT);
        } else if (allAdgroupsDraft) {
            return getCampaignSelfStatus(campaignIsolatedReasons);
        } else if (states.contains(CampaignStatesEnum.SUSPENDED)) {
            if (states.contains(CampaignStatesEnum.UNARCHIVING)) {
                return status(STOP_WARN, GdSelfStatusReason.CAMPAIGN_UNARCHIVING_IN_PROGRESS);
            } else {
                return status(STOP_OK, GdSelfStatusReason.CAMPAIGN_SUSPENDED_BY_USER);
            }
        } else if (states.contains(CampaignStatesEnum.UNARCHIVING)) {
            return null; // legit status "in progress"
        } else if (allAdgroupsSuspended) {
            return status(STOP_OK, GdSelfStatusReason.CAMPAIGN_ALL_ADGROUPS_SUSPENDED_BY_USER);
        } else if (allAdgroupsDraftOrSuspended) {
            return getCampaignSelfStatus(campaignIsolatedReasons);
        } else if (states.contains(CampaignStatesEnum.UNITS_EXHAUSTED)) {
            return status(STOP_OK, GdSelfStatusReason.CAMPAIGN_UNITS_EXHAUSTED);
        } else if (!hasActiveAdgroups) {
            if (hasAdGroupsOnModeration) {
                return getModerationStatus(cpmPriceReason == null ?
                        List.of(GdSelfStatusReason.CAMPAIGN_ON_MODERATION) :
                        List.of(cpmPriceReason, GdSelfStatusReason.CAMPAIGN_ON_MODERATION));
            } else {
                var reasons = new LinkedList<GdSelfStatusReason>();
                if (hasProcessingAdGroups) {
                    reasons.addFirst(allStoppedByBlProcessing
                            ? GdSelfStatusReason.CAMPAIGN_BL_PROCESSING
                            : GdSelfStatusReason.CAMPAIGN_HAS_PROCESSING_ADGROUPS);
                    if (!allAdGroupsProcessing) {
                        reasons.addFirst(GdSelfStatusReason.CAMPAIGN_HAS_INACTIVE_BANNERS);
                    }
                } else {
                    reasons.addFirst(allStoppedByNothingGenerated
                            ? GdSelfStatusReason.CAMPAIGN_BL_NOTHING_GENERATED
                            : GdSelfStatusReason.CAMPAIGN_HAS_NO_ADS_ELIGIBLE_FOR_SERVING);
                }
                return status(allAdgroupsOk ? STOP_PROCESSING : STOP_CRIT, reasons);
            }
        } else if (cpmPriceReason != null) {
            return status(STOP_CRIT, cpmPriceReason);
        } else if (states.contains(CampaignStatesEnum.NO_MONEY)) {
            GdSelfStatusReason needModeyReason = hasWallet ? GdSelfStatusReason.CAMPAIGN_ADD_MONEY_TO_WALLET
                    : GdSelfStatusReason.CAMPAIGN_ADD_MONEY;
            GdSelfStatusReason reason = states.contains(CampaignStatesEnum.AWAIT_PAYMENT)
                    ? GdSelfStatusReason.CAMPAIGN_WAIT_PAYMENT
                    : needModeyReason;
            return status(STOP_CRIT, unshiftAndCopyReasons(campaignIsolatedReasons, reason));
            // Временно убираем для всех кампаний: DIRECT-140177
            // } else if (states.contains(CampaignStatesEnum.PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_FUNDS)) {
            //     return status(PAUSE_WARN, PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_FUNDS);
        } else if (states.contains(CampaignStatesEnum.PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_CONVERSION)) {
            return status(RUN_WARN, GdSelfStatusReason.PAY_FOR_CONVERSION_CAMPAIGN_HAS_LACK_OF_CONVERSION);
        } else if (allAdgroupsActive && campaignPayed) {
            if (hasWarnedAdGroups) {
                return status(RUN_WARN, GdSelfStatusReason.CAMPAIGN_HAS_ADGROUPS_WITH_WARNINGS);
            } else if (hasProcessingAdGroups) {
                return status(RUN_PROCESSING, GdSelfStatusReason.CAMPAIGN_HAS_PROCESSING_ADGROUPS);
            } else if (hasPromoExtensionRejected) {
                return status(RUN_WARN, GdSelfStatusReason.PROMO_EXTENSION_REJECTED);
            }
            return status(RUN_OK);
        } else if (allAdgroupsOk && campaignPayed) {
            if (hasProcessingAdGroups) {
                return status(RUN_PROCESSING, GdSelfStatusReason.CAMPAIGN_HAS_PROCESSING_ADGROUPS);
            } else if (hasPromoExtensionRejected) {
                return status(RUN_WARN, GdSelfStatusReason.PROMO_EXTENSION_REJECTED);
            }
            return status(RUN_OK);
        } else if (campaignPayed) {
            // есть активные группы с проблемами и неактивные группы
            return hasWarnedAdGroups ? status(RUN_WARN, List.of(GdSelfStatusReason.CAMPAIGN_HAS_ADGROUPS_WITH_WARNINGS,
                    GdSelfStatusReason.CAMPAIGN_HAS_INACTIVE_BANNERS))
                    // есть неактивные группы
                    : status(RUN_WARN, GdSelfStatusReason.CAMPAIGN_HAS_INACTIVE_BANNERS);
        } else {
            logger.error("Error while calculating selfStatus for cid: {}", cid);
            return null; // какое-то не предусмотренное состояние
        }
    }

    private static SelfStatus getCampaignSelfStatus(List<GdSelfStatusReason> campaignIsolatedReasons) {
        if (campaignIsolatedReasons.isEmpty()) {
            return status(DRAFT);
        }
        return campaignIsolatedReasons.contains(GdSelfStatusReason.CAMPAIGN_ON_MODERATION)
                ? getModerationStatus(campaignIsolatedReasons)
                : status(DRAFT, campaignIsolatedReasons);
    }

    private static SelfStatus getModerationStatus(List<GdSelfStatusReason> reasons) {
        if (reasons.contains(GdSelfStatusReason.CPM_PRICE_NOT_APPROVED)) {
            return status(STOP_CRIT, reasons);
        } else {
            return status(ON_MODERATION, reasons);
        }
    }

    private static GdSelfStatusReason cpmPriceReasons(Collection<CampaignStatesEnum> states) {
        if (states.contains(CampaignStatesEnum.CPM_PRICE_WAITING_FOR_APPROVE)) {
            return GdSelfStatusReason.CPM_PRICE_WAITING_FOR_APPROVE;
        } else if (states.contains(CampaignStatesEnum.CPM_PRICE_INCORRECT)) {
            return GdSelfStatusReason.CPM_PRICE_INCORRECT;
        } else if (states.contains(CampaignStatesEnum.CPM_PRICE_NOT_APPROVED)) {
            return GdSelfStatusReason.CPM_PRICE_NOT_APPROVED;
        } else {
            return null;
        }
    }

    private static List<GdSelfStatusReason> unshiftAndCopyReasons(List<GdSelfStatusReason> reasons,
                                                                  GdSelfStatusReason addAsFirst) {
        if (reasons.isEmpty()) {
            return Collections.singletonList(addAsFirst);
        }
        List<GdSelfStatusReason> newReasons = new ArrayList<>();
        newReasons.add(addAsFirst);
        newReasons.addAll(reasons);
        return newReasons;
    }
}
