package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectOnConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupBsTagsRepository;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupTagsRepository;
import ru.yandex.direct.core.entity.adgroup.repository.typesupport.AdGroupTypeSupportDispatcher;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bids.container.BidDynamicOpt;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsDynamicStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsPerformanceStatusblgenerated;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.model.BannerGeoLegalFlags.VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.AGGR_STATUSES_ADGROUPS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbutil.SqlUtils.findInSet;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesAdGroupRepository {
    private final DslContextProvider ppcDslContextProvider;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final JooqReaderWithSupplier<AggregatedStatusAdGroup> adGroupJooqReader;
    private Set<Field<?>> fieldsToReadWithStatuses;
    private Set<Field<?>> fieldsToRead;

    @Autowired
    public AggregatedStatusesAdGroupRepository(DslContextProvider ppcDslContextProvider,
                                               AdGroupTagsRepository adGroupTagsRepository,
                                               AdGroupBsTagsRepository adGroupBsTagsRepository,
                                               BidModifierRepository bidModifierRepository,
                                               MinusKeywordsPackRepository minusKeywordsPackRepository,
                                               UsersSegmentRepository usersSegmentRepository,
                                               AdGroupTypeSupportDispatcher adGroupTypeSupportDispatcher,
                                               ShardHelper shardHelper,
                                               AggregatedStatusesRepository aggregatedStatusesRepository) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;

        adGroupJooqReader = JooqReaderWithSupplierBuilder.builder(AggregatedStatusAdGroup::new)
                .readProperty(AggregatedStatusAdGroup.ID, fromField(PHRASES.PID))
                .readProperty(AggregatedStatusAdGroup.CAMPAIGN_ID, fromField(PHRASES.CID))
                .readProperty(AggregatedStatusAdGroup.TYPE, fromField(PHRASES.ADGROUP_TYPE).by(AdGroupType::fromSource))
                .readProperty(AggregatedStatusAdGroup.STATUS_MODERATE, fromField(PHRASES.STATUS_MODERATE)
                        .by(StatusModerate::fromSource))
                .readProperty(AggregatedStatusAdGroup.BS_RARELY_LOADED, fromField(PHRASES.IS_BS_RARELY_LOADED)
                        .by(RepositoryUtils::booleanFromLong))
                .readProperty(AggregatedStatusAdGroup.GEO, fromField(PHRASES.GEO).by(AdGroupMappings::geoFromDb))
                .readProperty(AggregatedStatusAdGroup.START_TIME, fromField(ADGROUPS_INTERNAL.START_TIME))
                .readProperty(AggregatedStatusAdGroup.FINISH_TIME, fromField(ADGROUPS_INTERNAL.FINISH_TIME))
                .build();

        fieldsToRead = adGroupJooqReader.getFieldsToRead();

        fieldsToReadWithStatuses = new HashSet<>(adGroupJooqReader.getFieldsToRead());
        fieldsToReadWithStatuses.add(AGGR_STATUSES_ADGROUPS.AGGR_DATA);
        fieldsToReadWithStatuses.add(AGGR_STATUSES_ADGROUPS.IS_OBSOLETE);
    }


    /**
     * Получает список групп по списку их id, отсортированный в порядке возрастания id
     *
     * @param shard      шард
     * @param adGroupIds список id извлекаемых групп
     * @return список групп
     */
    public List<AggregatedStatusAdGroup> getAdGroups(int shard, Collection<Long> adGroupIds) {
        return selectAdGroups(shard)
                .where(PHRASES.PID.in(adGroupIds))
                .orderBy(PHRASES.PID)
                .fetch(adGroupJooqReader::fromDb);
    }

    /**
     * Получает соответствие групп со статусами к их id
     *
     * @param shard      шард
     * @param adGroupIds список id извлекаемых групп
     * @return Map где ключ — id, а значение — группа со статусом
     */
    public Map<Long, AggregatedStatusAdGroup> getAdGroupsWithStatusById(int shard, Collection<Long> adGroupIds) {
        if (CollectionUtils.isEmpty(adGroupIds)) {
            return Map.of();
        }
        return selectAdGroupsWithStatuses(shard)
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, this::fromDb);
    }

    /**
     * Получает список групп с аггрегированными статусоами по списку их id, отсортированный в порядке возрастания id
     * Группы с незаполненым статусом не возвращаются
     *
     * @param shard      шард
     * @param adGroupIds список id извлекаемых групп
     * @return список групп
     */
    public List<AggregatedStatusAdGroup> getAdGroupsWithStatusesForView(int shard, Collection<Long> adGroupIds) {
        return selectAdGroupsWithStatuses(shard)
                .where(PHRASES.PID.in(adGroupIds))
                .orderBy(PHRASES.PID)
                .fetch(this::fromDb);
    }

    public List<AggregatedStatusAdGroup> getAdGroupsWithStatusesByCidForPopup(int shard, ClientId clientId,
                                                                              Long campaignId) {
        return selectAdGroupsWithStatusesAndCampaigns(shard)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()).and(PHRASES.CID.eq(campaignId)))
                .orderBy(PHRASES.PID)
                .fetch(this::fromDb);
    }

    private AggregatedStatusAdGroup fromDb(Record record) {
        AggregatedStatusAdGroup adgroup = adGroupJooqReader.fromDb(record);
        String aggrStatusJson = record.get(AGGR_STATUSES_ADGROUPS.AGGR_DATA);
        Boolean isObsolete = RepositoryUtils.booleanFromLong(record.get(AGGR_STATUSES_ADGROUPS.IS_OBSOLETE));
        AggregatedStatusAdGroupData aggrStatus = aggrStatusJson != null
                ? aggregatedStatusesRepository.fromJsonSafe(adgroup.getId(), aggrStatusJson,
                AggregatedStatusAdGroupData.class).withIsObsolete(isObsolete)
                : null;
        adgroup.setAggregatedStatus(aggrStatus);
        return adgroup;
    }

    /**
     * список групп по списку id кампаний
     *
     * @param shard
     * @param campaignIds
     * @return
     */
    public List<AggregatedStatusAdGroup> getAdGroupsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return selectAdGroups(shard)
                .where(PHRASES.CID.in(campaignIds))
                .orderBy(PHRASES.PID)
                .fetch(adGroupJooqReader::fromDb);
    }

    public List<Long> getAdGroupIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(PHRASES.PID)
                .from(PHRASES)
                .where(PHRASES.CID.in(campaignIds))
                .fetch(PHRASES.PID);
    }

    public List<Long> getCampaignIdsByAdGroupIds(int shard, Collection<Long> adgroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(PHRASES.CID)
                .from(PHRASES)
                .where(PHRASES.PID.in(adgroupIds))
                .fetch(PHRASES.CID);
    }

    /**
     * Получаем группы у которых есть таргетинг по интересам
     * интересы - условие показа, не имеющее собственного статуса. Для подсчета статуса на группе
     * нам важно лишь знать есть ли на ней таргетинг по интересам или нет
     *
     * @param shard      шард
     * @param adGroupIds список id групп
     * @return
     */
    public Set<Long> getAdGroupIdsWithInterests(int shard, Collection<Long> adGroupIds) {
        // выбираем только интересы, все остальное в гридах имеет свой статус, который надо будет вычислять
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(BIDS_RETARGETING.PID)
                .from(BIDS_RETARGETING)
                .join(RETARGETING_CONDITIONS).using(BIDS_RETARGETING.RET_COND_ID)
                .where(BIDS_RETARGETING.PID.in(adGroupIds)
                        .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE
                                .eq(RetargetingConditionsRetargetingConditionsType.interests)))
                .and(BIDS_RETARGETING.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(Boolean.FALSE)))
                .fetchSet(BIDS_RETARGETING.PID);
    }

    /**
     * Получаем группы в которых есть баннеры с флагами, требующими предоставления лиценизии
     *
     * @param shard      шард
     * @param adGroupIds список id групп
     * @return
     */
    public Set<Long> getAdGroupIdsWithGeoLegalFlags(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptySet();
        }
        var rows = ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.PID, BANNERS.FLAGS)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .and(BANNERS.FLAGS.isNotNull())
                .and(BANNERS.FLAGS.ne(""))
                .fetch();
        return rows.stream().filter(r ->
                        BannerFlags.fromSource(r.get(BANNERS.FLAGS)).getFlags().keySet()
                                .stream().anyMatch(VALUES::contains))
                .map(r -> r.get(BANNERS.PID))
                .collect(Collectors.toSet());
    }

    /**
     * Возвращает id групп из списка {@param adGroupIds}, у которых STATUS_BL_GENERATED=Processing
     */
    public Map<Long, StatusBLGenerated> getGroupStatusBlGenerated(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, StatusBLGenerated> result = new HashMap<>();

        ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID, ADGROUPS_DYNAMIC.STATUS_BL_GENERATED, ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED)
                .from(PHRASES)
                .leftJoin(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .leftJoin(ADGROUPS_PERFORMANCE).on(ADGROUPS_PERFORMANCE.PID.eq(PHRASES.PID))
                .where(PHRASES.PID.in(adGroupIds))
                .fetch()
                .forEach(r -> {
                    Long pid = r.get(PHRASES.PID);
                    StatusBLGenerated status =
                            dynOrPerfBlGeneratedFromDb(r.get(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED),
                                    r.get(ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED));
                    if (status != null) {
                        result.put(pid, status);
                    }
                });

        return result;
    }

    private StatusBLGenerated dynOrPerfBlGeneratedFromDb(@Nullable AdgroupsDynamicStatusblgenerated dynamic,
                                                         @Nullable AdgroupsPerformanceStatusblgenerated performance) {
        if (dynamic == null) {
            return StatusBLGenerated.fromSource(performance);
        }

        // да это не красиво, но везде делать ветку dynamic/performance в то время как BlGeneratedStatus
        // должен быть одинаковым, кажется излишним
        switch (dynamic) {
            case No:
                return StatusBLGenerated.NO;
            case Processing:
                return StatusBLGenerated.PROCESSING;
            case Yes:
                return StatusBLGenerated.YES;
            default:
                return null;
        }
    }

    /**
     * возвращает Map adgroupId и ShowConditionCount, где ShowConditionCount это счетчик условий показа для типов
     * групп у которых условия показа не имеют своих статусов, но считать их надо более сложно нежели просто на наличие
     *
     * @param shard
     * @param adGroupIds
     * @return
     */
    public Map<Long, ShowConditionsCounter> getAdGroupsShowConditionCounts(int shard, Collection<Long> adGroupIds) {
        // выбираем только интересы, все остальное в гридах имеет свой статус, который надо будет вычислять
        Map<Long, ShowConditionsCounter> result = new HashMap<>();
        Field<Integer> isSuspendedField = DSL.iif(findInSet(BidDynamicOpt.SUSPENDED.getTypedValue(), BIDS_DYNAMIC.OPTS)
                .ge(1L), 1, 0).as("dynamic_suspended");
        var rows = ppcDslContextProvider.ppc(shard)
                .select(PHRASES.PID,
                        BIDS_PERFORMANCE.PERF_FILTER_ID, BIDS_PERFORMANCE.IS_SUSPENDED,
                        BIDS_DYNAMIC.DYN_ID, isSuspendedField)
                .from(PHRASES)
                .leftJoin(BIDS_PERFORMANCE)
                .on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID).and(BIDS_PERFORMANCE.IS_DELETED.ne(1L)))
                .leftJoin(BIDS_DYNAMIC).on(PHRASES.PID.eq(BIDS_DYNAMIC.PID))
                .where(PHRASES.ADGROUP_TYPE.in(PhrasesAdgroupType.performance, PhrasesAdgroupType.dynamic)
                        .and(PHRASES.PID.in(adGroupIds)))
                .fetch();

        for (var r : rows) {
            var counter = result.computeIfAbsent(r.get(PHRASES.PID), id -> new ShowConditionsCounter());
            if (r.get(BIDS_PERFORMANCE.PERF_FILTER_ID) != null) {
                counter.incrTotal();
                if (longContainsTrue(r.get(BIDS_PERFORMANCE.IS_SUSPENDED))) {
                    counter.incrSuspended();
                }
            }
            if (r.get(BIDS_DYNAMIC.DYN_ID) != null) {
                counter.incrTotal();
                if (r.get(isSuspendedField) == 1) {
                    counter.incrSuspended();
                }
            }
        }

        return result;
    }

    private boolean longContainsTrue(Long flag) {
        return Boolean.TRUE.equals(RepositoryUtils.booleanFromLong(flag));
    }

    private SelectOnConditionStep<Record> selectAdGroups(int shard) {
        return selectAdGroups(shard, fieldsToRead);
    }

    private SelectOnConditionStep<Record> selectAdGroupsWithStatuses(int shard) {
        return selectAdGroups(shard, fieldsToReadWithStatuses)
                .join(AGGR_STATUSES_ADGROUPS).on(PHRASES.PID.eq(AGGR_STATUSES_ADGROUPS.PID));
    }

    private SelectOnConditionStep<Record> selectAdGroupsWithStatusesAndCampaigns(int shard) {
        return selectAdGroupsWithStatuses(shard)
                .join(CAMPAIGNS).on(PHRASES.CID.eq(CAMPAIGNS.CID));
    }

    private SelectOnConditionStep<Record> selectAdGroups(int shard, Set<Field<?>> fieldsToRead) {
        return ppcDslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(PHRASES)
                .leftJoin(ADGROUPS_INTERNAL).on(PHRASES.PID.eq(ADGROUPS_INTERNAL.PID));
    }

}
