package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Row3;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.aggregatedstatuses.ChangesHolder;
import ru.yandex.direct.core.entity.keyword.aggrstatus.StatusAggregationArchivedKeyword;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.keywordChangesEvent;
import static ru.yandex.direct.dbschema.ppc.tables.BidsArc.BIDS_ARC;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesArchivedKeywordRepository {

    private final DslContextProvider ppcDslContextProvider;
    private final Collection<Field<?>> fields;
    private final JooqReaderWithSupplier<StatusAggregationArchivedKeyword> reader;

    @Autowired
    public AggregatedStatusesArchivedKeywordRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;

        reader = JooqReaderWithSupplierBuilder.builder(StatusAggregationArchivedKeyword::new)
                .readProperty(StatusAggregationArchivedKeyword.ID, fromField(BIDS_ARC.ID))
                .readProperty(StatusAggregationArchivedKeyword.AD_GROUP_ID, fromField(BIDS_ARC.PID))
                .readProperty(StatusAggregationArchivedKeyword.CAMPAIGN_ID, fromField(BIDS_ARC.CID))
                .build();

        fields = this.reader.getFieldsToRead();
    }

    public Set<Long> getArchivedKeywordsForStatusAggregationByPKeyTriplets(
            int shard, Collection<CidPidIdTriplet> cidPidIdTriplets) {
        List<Row3<Long, Long, Long>> cidPidIdRows = cidPidIdTriplets.stream()
                .map(pkey -> row(pkey.getCid(), pkey.getPid(), pkey.getId())).collect(Collectors.toList());

        return ppcDslContextProvider.ppc(shard)
                .select(fields)
                .from(BIDS_ARC)
                .where(row(BIDS_ARC.CID, BIDS_ARC.PID, BIDS_ARC.ID).in(cidPidIdRows))
                .fetchSet(BIDS_ARC.ID);
    }

    public static class CidPidIdTriplet {
        private final long cid;
        private final long pid;
        private final long id;

        public CidPidIdTriplet(long cid, long pid, long id) {
            this.cid = cid;
            this.pid = pid;
            this.id = id;
        }

        public long getCid() {
            return cid;
        }

        public long getPid() {
            return pid;
        }

        public long getId() {
            return id;
        }
    }

    private ChangesHolder.KeywordChangesEvent toKeywordChangesEvent(Record record) {
        return keywordChangesEvent(record.get(BIDS_ARC.ID), record.get(BIDS_ARC.PID), record.get(BIDS_ARC.CID), false);
    }
}
